function [ G, M, idx ] = threshold_by_sparsity( M, sparsity )

%THRESHOLD_BY_SPARSITY Thresholds a weighted matrix M such that the
% resulting graph has the given SPARSITY (as a proportion from 0 to 1); or 
% at least a sparsity closest to this value without arbitraily removing
% identical values. For example, if many values are similar around the
% threshold level, none or all of these values are thresholded (determined
% by the 'smaller' argument).
% 
% PARAMETERS:
%   M           An N x N weighted matrix, 
%   SPARSITY    The desired sparsity, as a proportion from 0 (no edges) to
%               1 (all edges).
%
% RETURNS:
%   G - A binarized graph where G(i,j)= 1 indicates the existence of an edge 
%       between vertices i and j.
%   M - A weighted graph, thresholded by G
%

M = M - diag(diag(M)); % Remove diagonal
N = length(M);
K = round(sparsity*(N)*(N-1));
if K == 0
   G = false(N);
   M = M.*double(G);
   return;
end

M_tmp = reshape(M, N*N, 1);
[M_sort M_ind] = sort(M_tmp, 'descend');      % Sort elements
val = M_sort(K);
if val > 0
    G = M >= val;
else
    G = M > val;
end

G = G - diag(diag(G)); % Remove diagonal again
M = M.*double(G);
idx = find(G);

end

