function [  ] = subject_edge_residuals_glmnet2pajek( cv_result, betas, output_path_prefix, threshold, split_pos_neg, idx, vertex_vals, varargin )
%SUBJECT_EDGE_RESIDUAL_GLMNET2PAJEK 
%
% Writes subject edge residual predictions obtained through glmnet (e.g. 
% the output of @analyze_subject_edge_residuals_glmnet) as a Pajek graph.
% Uses the optimal lambda value (lambda1se), unless the desired index is
% specified by the idx parameter. Uses the beta values from this lambda as edge
% weights.
%
% PARAMETERS:
% 
% cv_result             A result from cvglmnet; i.e., for a single alpha value
% betas                 Beta values to use as edge weights (typically derived from
%                       cv_result)
% output_path_prefix    Path and prefix for output file; suffix and
%                       extension (.net) will be appended to this
% threshold             Threshold for (absolute) weight values
% split_pos_neg         Whether to split the weights into two files, one for
%                       positive betas and one for negative (which are
%                       converted to positive); default is true
% idx                   [Optional] the index from which to obtain beta values
% vertex_vals           Values to assign to graph vertices; one of 'max', 
%                       'sum', 'mean', 'all', or 'none'; default is 'none'; output
%                       will be written to a separate text file
% varargin              Arguments to the write_matrix_to_pajek function
%
% AUTHOR: Andrew Reid, MNI, 2012
%

glmnet_object = cv_result.glmnet_object;

if nargin < 4
    threshold = NaN;
end

if nargin < 5
    split_pos_neg = true;
end

if nargin < 6 || idx < 1
   lambda =  cv_result.lambda_1se;
   idx=find(glmnet_object.lambda==lambda);
   if numel(idx)==0
      error(['Optimal lambda ', num2str(lambda), ' not found in list..']);
   end
end

if nargin < 7
    vertex_vals = 'none';
end

if length(size(betas)) == 3
    beta_opt = betas(:,:,idx);
else
    beta_opt = betas;
    N=sqrt(numel(beta_opt));
    beta_opt=reshape(beta_opt,N,N);
end
if ~isnan(threshold)
    beta_opt=beta_opt.*(abs(beta_opt)>threshold);
end

N=length(beta_opt);

if strcmp(vertex_vals, 'max')
    beta_vertex = max(beta_opt);
elseif strcmp(vertex_vals, 'min')
    beta_vertex = min(beta_opt);
elseif strcmp(vertex_vals, 'abs_max')
    beta_vertex = max(abs(beta_opt));
elseif strcmp(vertex_vals, 'sum')
    beta_vertex = sum(beta_opt);
elseif strcmp(vertex_vals, 'pos_sum')
    beta_vertex = sum(beta_opt.*(beta_opt>0));
elseif strcmp(vertex_vals, 'neg_sum')
    beta_vertex = sum(beta_opt.*(beta_opt<0));
elseif strcmp(vertex_vals, 'abs_sum')
    beta_vertex = sum(abs(beta_opt));
elseif strcmp(vertex_vals, 'mean')
    beta_vertex = mean(beta_opt,1);
elseif strcmp(vertex_vals, 'pos_mean')
    beta_vertex = mean(beta_opt.*(beta_opt>0),1);
elseif strcmp(vertex_vals, 'neg_mean')
    beta_vertex = mean(beta_opt.*(beta_opt<0),1);
elseif strcmp(vertex_vals, 'abs_mean')
    beta_vertex = mean(abs(beta_opt),1);
elseif strcmp(vertex_vals, 'count')
    beta_vertex = sum(abs(beta_opt)>0);
elseif strcmp(vertex_vals, 'all')
    beta_vertex=zeros(12,N);
    beta_vertex(1,:) = max(beta_opt);
    beta_vertex(2,:) = min(beta_opt);
    beta_vertex(3,:) = max(abs(beta_opt));
    beta_vertex(4,:) = sum(beta_opt);
    beta_vertex(5,:) = sum(beta_opt.*(beta_opt>0));
    beta_vertex(6,:) = sum(beta_opt.*(beta_opt<0));
    beta_vertex(7,:) = sum(abs(beta_opt));
    beta_vertex(8,:) = mean(beta_opt,1);
    beta_vertex(9,:) = mean(beta_opt.*(beta_opt>0),1);
    beta_vertex(10,:) = mean(beta_opt.*(beta_opt<0),1);
    beta_vertex(11,:) = mean(abs(beta_opt),1);
    beta_vertex(12,:) = sum(abs(beta_opt)>0);
    
end

if ~split_pos_neg
    write_matrix_to_pajek(beta_opt, [output_path_prefix, '.net'], varargin{:});
else
    beta_opt_pos=beta_opt.*double(beta_opt>0);
    write_matrix_to_pajek(beta_opt_pos, [output_path_prefix, '_pos.net'], varargin{:});
    beta_opt_neg=abs(beta_opt.*double(beta_opt<0));
    write_matrix_to_pajek(beta_opt_neg, [output_path_prefix, '_neg.net'], varargin{:});
end

if exist('beta_vertex', 'var')
    fname = [output_path_prefix, '_', vertex_vals, '.txt'];
    header = [{'max'},{'min'},{'abs_max'},{'sum'},{'pos_sum'},{'neg_sum'},{'abs_sum'}, ...
              {'mean'},{'pos_mean'},{'neg_mean'},{'abs_mean'},{'count'}];
    write_matrix(fname, beta_vertex, '\t', header);
end

end
