function [standardized] = standardize_data(Y, stdev, submean, pooled, rows)

% Standardizes Y by subtracting the mean and dividing by the standard deviation
% 
% Parameters:
% 
%     Y             A N x M matrix of data, where N is the number of subjects
%                   (independent data sources) and M is the number of data
%                   points per subject.
% 
%     stdev         Divide by the standard deviation (default = true)
%
%     submean       Subtract the mean (default = true)
%
%     pooled        If true, uses the pooled mean and standard deviation;
%                   otherwise uses M-wise values. Default = true.
%
%     rows          If true, standardizes by rows (default); otherwise by 
%                   columns.
% 
% Returns:
%
%     The standardized data 
%
% AUTHOR Andrew Reid, MNI, 2012

if nargin < 2
   stdev = true; 
end

if nargin < 3
   submean = true; 
end

if nargin < 4 || pooled == true
   p_stdev = std(Y(:)); 
   p_mean = mean(Y(:));
   if submean == true, standardized = Y - p_mean; end;
   if stdev == true, standardized = standardized./p_stdev; end;
   return;
end

if nargin < 5
   rows = true; 
end

if rows
    M = size(Y,2);
else
    M = size(Y,1); 
end
standardized = Y;

if rows
    l_stdev = std(Y,0,1);
    l_mean = mean(Y,1);
else
    l_stdev = std(Y,0,2);
    l_mean = mean(Y,2);
end

for i = 1:M

    if (rows)
        if submean == true
            standardized(:,i) = standardized(:,i) - l_mean(i);
        end
        if stdev == true
            standardized(:,i) = standardized(:,i)./l_stdev(i);
        end
    else
        if submean == true
            standardized(i,:) = standardized(i,:) - l_mean(i);
        end
        if stdev == true
            standardized(i,:) = standardized(i,:)./l_stdev(i);
        end
    end
end