function [ V_smooth ] = smooth_volume_gaussian( V, fwhm, max_radius, verbose )
% smooth_volume_gaussian Apply Gaussian smoothing to V
%   
% fwhm       - The full width at half max of the Gaussian (equal to 2 *
%             sqrt(2 * log2) * sigma)
% max_radius - The maximum radius, in voxel units, at which to sample values
% verbose    - If true, prints progress to screen
%

if nargin < 4
   verbose = false; 
end

N = size(V);
if length(N) ~= 3
   error('First parameter must be a 3-dimensional matrix!'); 
end

sigma = fwhm / (2 * sqrt(2 * log(2)));
V_smooth = zeros(size(V));

search = 1+round(max_radius*2);
dist = zeros(search,search,search);
clear XYZ;
[XYZ(:,1),XYZ(:,2),XYZ(:,3)] = ind2sub(size(dist),1:length(dist(:)));

% Euclidean distances for the search cube
for i = 1 : search
   for j = 1 : search
      for k = 1 : search
         delta_x = max_radius - i;
         delta_y = max_radius - j; 
         delta_z = max_radius - k; 
         dist(i,j,k) = sqrt(delta_x * delta_x + ...
                            delta_y * delta_y + ...
                            delta_z * delta_z);
      end
   end
end

% Slice image
V_slice = cell(N(1),1);
accum = 0;

parfor i = 1 : N(1)
    V_i = zeros(N(2),N(3));
    for j = 1 : N(2)
        for k = 1 : N(3)
            min_i = max(1,i-max_radius);
            max_i = min(N(1),i+max_radius);
            min_j = max(1,j-max_radius);
            max_j = min(N(2),j+max_radius);
            min_k = max(1,k-max_radius);
            max_k = min(N(3),k+max_radius);
            nbrs = V(min_i:max_i, ...
                     min_j:max_j, ...
                     min_k:max_k);

            if sum(nbrs(:))>0
                summ = 0;
                denom = 0;
                for ii = i-max_radius : i+max_radius
                   if ii > 1 && ii <= N(1)
                       si = ii - (i-max_radius) + 1;
                       for jj = j-max_radius : j+max_radius
                          if jj > 1 && jj <= N(2)
                              sj = j - (j-max_radius) + 1;
                              for kk = k-max_radius : k+max_radius
                                  if kk > 1 && kk <= N(3)
                                      sk = k - (k-max_radius) + 1;
                                      % Do your thing
                                      d = dist(si,sj,sk);
                                      w = normpdf(d,0,sigma);
                                      v = V(ii,jj,kk);
                                      summ = summ + v * w;
                                      denom = denom + w;
                                  end
                              end
                          end
                       end
                   end
                end
                V_i(j,k) = summ / denom;
                %V_smooth(i,j,k) = summ / denom;
            end
        end
    end
    V_slice(i) = {V_i};
    disp(sprintf('Done slice %d\n', i));
end

% Put slices together
for i = 1 : N(1)
    V_smooth(i,:,:) = V_slice{i};
end

