function results = roi_multiple_regression3(Y, R, include_main)

% roi_multiple_regression
%
% Performs a multiple linear regression on each pair of ROIs in an N x m
% matrix R, where N is the number of subjects and m is the number of ROIs.
% The "Y" vector for the regression is equal to R(:,i) * R(:,j), and the
% factors are specified by the N x f matrix F, where f = the number of
% factors to regress.
%
% This alternative form of the method uses the Matlab function @regstats,
% which computes t statistics for each factor, along with the overall
% F stats. It does not compute confidence intervals however.
%
% Usage:    results = roi_multiple_regression(R, F);
%
% Parameters:
%
%   R:          N x m matrix of values (e.g., ROI-wise mean thickness, or 
%               residuals) where N is the number of subjects and m is the 
%               number of values (e.g., ROIs) per subject.
%   Y:          N x 1 variable to predict.
%   include_main:   Specifies whether to include main effect terms in the 
%   [optional]      regression model (R(:,i) and R(I,j)).
%                  
%   
% Returns:
%
%   result: A struct containing:
%
%             r_square:     The R-squared value for the model at i,j
%             adj_r_square: Adjusted R-squared value
%             beta:         Coefficients for the factors in the model, with
%                           the last vector corresponding to constant term,
%                           and the two preceeding corresponding to the
%                           main effects terms, if these were included.
%             f:            F-values for the model at i,j.
%             f_pval:       Corresponding p-values. 
%             f_fdr:        FDR thresholded at q=0.05 for each F model at
%                           i,j.
%             t:            T-values for each factor in the model at i,j.
%             t_pval:       Corresponding p-values.
%             t_fdr:        FDR thresholded at 0.05 for each T model.
%             err:          The error term for the model at i,j
% 
% Author:   Andrew Reid, 2011, BIC, MNI
%
% See also: regress
%        

if nargin < 3
    include_main = false;
end 

N = size(R,1);
m = size(R,2);

if (N ~= size(Y,1))
   fprintf('\nR and Y matrices dont match!');
   return;
end

f = 1; % size(F,2);
if include_main == true
   f=3;
end 

results.r_square = zeros(m,m);
results.adj_r_square = zeros(m,m);
results.beta = zeros(m,m,f+1);
results.f_pval = zeros(m,m);
results.f = zeros(m,m);
results.t = zeros(m,m,f+1);
results.t_pval = zeros(m,m,f+1);
results.err = zeros(m,m);

% Determines which stats to compute with regstats (type 'help regstats')
whichstats={'beta','rsquare','adjrsquare','tstat','fstat'};

% For each ROI pair
for i = 1:m
    
    % everything before i has already been done
    for j = 1:m
        
        if i ~= j
            % Calculate regression model for i,j
            factors = R(:,i).*R(:,j);
            if include_main == true
                factors = [R(:,i), R(:,j)];
                stats=regstats(Y, factors, 'interaction', whichstats);
            else
                stats=regstats(Y, factors, 'linear', whichstats);
            end
            
            results.r_square(i,j) = stats.rsquare;
            results.adj_r_square(i,j) = stats.adjrsquare;
            results.beta(i,j,:) = stats.beta';
            results.f(i,j) = stats.fstat.f;
            results.f_pval(i,j) = stats.fstat.pval;
            results.t(i,j,:) = stats.tstat.t;
            results.t_pval(i,j,:) = stats.tstat.pval;
            
%             results.r_square(j,i) = stats.rsquare;
%             results.adj_r_square(j,i) = stats.adjrsquare;
%             results.beta(j,i,:) = stats.beta';
%             results.f(j,i) = stats.fstat.f;
%             results.f_pval(j,i) = stats.fstat.pval;
%             results.t(j,i,:) = stats.tstat.t;
%             results.t_pval(j,i,:) = stats.tstat.pval;
            
        end
        
    end
    
    fprintf('.');
    
end

% Run FDR
results.f_fdr=fdr_bky(results.f_pval,0.05,'no');

for i = 1:f+1
    results.t_fdr(:,:,i)=fdr_bky(results.t_pval(:,:,i),0.05,'no');
end

fprintf('\nDone.\n');