function results = roi_multiple_regression2(R, F, show_figures, figure_title, factor_names)

% roi_multiple_regression
%
% Performs a multiple linear regression on each pair of ROIs in an N x m
% matrix R, where N is the number of subjects and m is the number of ROIs.
% The "Y" vector for the regression is equal to R(:,i) * R(:,j), and the
% factors are specified by the N x f matrix F, where f = the number of
% factors to regress.
%
% This alternative form of the method uses the Matlab function @regstats,
% which computes t statistics for each factor, along with the overall
% F stats. It does not compute confidence intervals however.
%
% Usage:    results = roi_multiple_regression(R, F);
%
% Parameters:
%
%   R:          N x m matrix of values (e.g., ROI-wise mean thickness, or 
%               residuals) where N is the number of subjects and m is the 
%               number of values (e.g., ROIs) per subject.
%   F:          N x f matrix of regressors; i.e., factors to use in the linear
%               regression model per ROI pair. f is the number of factors. Note
%               that F should NOT include a constant term (will be added
%               automatically).                  
%   include_main:   Include main terms R(:,i) and (R:,j) in the model
%
% Returns:
%
%   result: A struct containing:
%
%             r_square:     The R-squared value for the model at i,j
%             adj_r_square: Adjusted R-squared value
%             beta:         Coefficients for the factors in the model, with
%                           the last vector corresponding to constant term,
%                           and the two preceeding corresponding to the
%                           main effects terms, if these were included.
%             f:            F-values for the model at i,j.
%             f_pval:       Corresponding p-values. 
%             f_fdr:        FDR thresholded at q=0.05 for each F model at
%                           i,j.
%             t:            T-values for each factor in the model at i,j.
%             t_pval:       Corresponding p-values.
%             t_fdr:        FDR thresholded at 0.05 for each T model.
%             err:          The error term for the model at i,j
% 
% Author:   Andrew Reid, 2011, BIC, MNI
%
% See also: regress
%        

% if nargin < 3
%     include_main = false;
% end 

if nargin < 4
   has_title=false;
else
   has_title=true;
end

if nargin < 5
   has_names=false;
else
   has_names=true;
end

N = size(R,1);
m = size(R,2);

if (N ~= size(F,1))
   fprintf('\nR and F matrices dont match!');
   return;
end

f = size(F,2);
% if include_main == true
%    f = f + 2; 
% end

results.r_square = zeros(m,m);
results.adj_r_square = zeros(m,m);
results.beta = zeros(m,m,f+1);
results.f_pval = zeros(m,m);
results.f = zeros(m,m);
results.t = zeros(m,m,f+1);
results.t_pval = zeros(m,m,f+1);
results.err = zeros(m,m);

% Determines which stats to compute with regstats (type 'help regstats')
whichstats={'beta','rsquare','adjrsquare','tstat','fstat'};

% For each ROI pair
for i = 1:m
    
    % everything before i has already been done
    for j = 1:m
        
        if i ~= j
            % Calculate regression model for i,j
            factors = F;
            stats=regstats(R(:,i).*R(:,j), factors, 'linear', whichstats);
            results.r_square(i,j) = stats.rsquare;
            results.adj_r_square(i,j) = stats.adjrsquare;
            results.beta(i,j,:) = stats.beta;
            results.f(i,j) = stats.fstat.f;
            results.f_pval(i,j) = stats.fstat.pval;
            results.t(i,j,:) = stats.tstat.t;
            results.t_pval(i,j,:) = stats.tstat.pval;
            
        end
        
    end
    
    fprintf('.');
    
end

% Run FDR
results.f_fdr=fdr_bky(results.f_pval,0.05,'no');

for i = 1:f+1
    results.t_fdr(:,:,i)=fdr_bky(results.t_pval(:,:,i),0.05,'no');
end

if show_figures==true
    fid=figure;
    rows = 1 + floor((f + 1) / 4);
    if f + 1 > 4
        cols = 4;
    else
        cols = f + 1;
    end
    for i = 1:f+1
        sid=subplot(rows, cols, i);
        if i == 1
            caption = 'Constant term';
        else
            caption = cell2mat(strcat({'Term '},num2str(i)));
            if has_names
                test=length(factor_names);
                if i <= test + 1
                    caption = cell2mat(strcat(factor_names(i - 1), {' term'}));
                end
            end
        end    
       
         
        iid=image (results.beta(:,:,i).*double(results.t_pval(:,:,i)<0.05).*double(results.t_fdr(:,:,i)));
        this_title = cell2mat(strcat({figure_title},{' Sig betas'}));
        title(strcat({this_title caption}));
        set(iid,'CDataMapping','scaled');
        set(fid,'Color','white');
        
        colorbar;
    end
    
    
     % T-score images
    fid=figure;
    for i = 1:f+1
        sid=subplot(rows, cols, i);
        if i == 1
            caption = 'Constant term';
        else
            caption = cell2mat(strcat({'Term '},num2str(i)));
            if has_names
                test=length(factor_names);
                if i <= test + 1
                    caption = cell2mat(strcat(factor_names(i - 1), {' term'}));
                end
            end
        end    
        
        iid=image (results.t(:,:,i));
        this_title = cell2mat(strcat({figure_title},{' T-values'}));
        title({this_title caption});
        set(iid,'CDataMapping','scaled');
        set(fid,'Color','white');

        colorbar;
    end
    
    % R-squared
    fid=figure;
    sid=subplot(1, 2, 1);

    iid=image (results.r_square.*double(results.f_pval<0.05).*double(results.f_fdr));
    this_title = cell2mat(strcat({figure_title},{' R-squared'}));
    title(this_title);
    set(iid,'CDataMapping','scaled');
    set(fid,'Color','white');
    colorbar;
    
    sid=subplot(1, 2, 2);

    iid=image (results.adj_r_square.*double(results.f_pval<0.05).*double(results.f_fdr));
    this_title = cell2mat(strcat({figure_title},{' Adj-R-squared'}));
    title(this_title);
    set(iid,'CDataMapping','scaled');
    set(fid,'Color','white');
    colorbar;
    
    
end

fprintf('\nDone.\n');