function results = roi_multiple_regression(R, F)

% roi_multiple_regression
%
% Performs a multiple linear regression on each pair of ROIs in an N x m
% matrix R, where N is the number of subjects and m is the number of ROIs.
% The "Y" vector for the regression is equal to R(:,i) * R(:,j), and the
% factors are specified by the N x f matrix F, where f = the number of
% factors to regress.
%
% Usage:    results = roi_multiple_regression(R, F);
%
% Parameters:
%
%   R:      N x m matrix of values (e.g., ROI-wise mean thickness, or 
%           residuals) where N is the number of subjects and m is the 
%           number of values (e.g., ROIs) per subject.
%   F:      N x f matrix of regressors; i.e., factors to use in the linear
%           regression model per ROI pair. f is the number of factors. Note
%           that F should include a constant term (i.e., extra column of
%           1's).
%
% Returns:
%
%   result: An m x m matrix of results, each a struct containing:
%   
%             r_square:     The R-squared value for the model at i,j
%             r_int:        The confidence interval for R-square
%             b:            An f x 1 vector of b-values for each factor in the 
%                           model. 
%             b_int:        The confidence interval for the b-values
%             p:            The p-value for the model at i,j
%             f:            The f-value for the model at i,j
%             err:          The error term for the model at i,j
% 
% Author:   Andrew Reid, 2011
%
% See also: regress
%        

% For each ROI pair

N = size(R,1);
m = size(R,2);

if (N ~= size(F,1))
   error('\nR and F matrices dont match!');
end

f = size(F,2);

results.r_square = zeros(m,m);
results.b = zeros(m,m,f);
results.b_int = zeros(m,m,f*2);
results.p = zeros(m,m);
results.f = zeros(m,m);
results.err = zeros(m,m);

for i = 1:m
    
    % everything before i has already been done
    for j = i:m
        
        if i ~= j
            % Calculate regression model for i,j
            [b,bint,~,~,stats]=regress(R(:,i).*R(:,j), F);
            results.r_square(i,j) = stats(1);
            results.b(i,j,:) = b';
            test = [bint(:,1);bint(:,2)];
            results.b_int(i,j,:) = test;
            results.p(i,j) = stats(3);
            results.f(i,j) = stats(2);
            results.err(i,j) = stats(4);
            results.r_square(j,i) = stats(1);
            results.b(j,i,:) = b';
            test = [bint(:,1);bint(:,2)];
            results.b_int(j,i,:) = test;
            results.p(j,i) = stats(3);
            results.f(j,i) = stats(2);
            results.err(j,i) = stats(4);
        end
        
    end
    
    fprintf('.');
    
end

fprintf('\nDone.\n');