function [ Y, subjects, rejected ] = read_civet_data( civet_dir, subdir, prefix, pattern, hemi, subjects )
%READ_CIVET_DATA 
%
% Reads vertex-wise data (e.g., cortical thickness values) for left, right, or both hemispheres
%
% PARAMETERS
% 
% civet_dir         The civet output directory
% subdir            The subdirectory from which to obtain data (e.g.,
%                   'thickness')
% subjects          A list of subjects to read; default is all subjects
% hemi              Hemisphere to read; choice of 'left', 'right', or
%                   'both'; default is 'both'
% prefix            Prefix for file name (inserts before subject id)
% pattern           File pattern to match; this inserts after the subject 
%                   id and before the suffix + extension (e.g., '_left.txt' 
%                   or '_right.txt')
%
% AUTHOR Andrew Reid, MNI, 2012

if nargin < 5
   hemi = 'both'; 
end

subject_list = dir(civet_dir);
if nargin < 6
    isdir = [subject_list.isdir];
    subjects = {subject_list.name};
    subjects = subjects(isdir);
    subjects = subjects(3:end);
end

j = 1;
subs=cell(0);
% Filter subject list
for i = 3 : numel(subject_list)
   
    this_subject = subject_list(i);
    
    if (this_subject.isdir)
        if nargin < 5
           subs(j) = {this_subject.name};
           j = j + 1;
        else
           test = strcmp(this_subject.name,subjects);
           if sum(test)>0
               subs(j) = {this_subject.name};
               j = j + 1;
           end
        end
    end
    
end

if numel(subs) == 0
   error('No subjects found!'); 
end

% Build file names
if strcmp(hemi,'both')
   thick_files=cell(numel(subs),2); 
else
   thick_files=cell(numel(subs),1); 
end

to_keep=true(numel(subs),1);

for i = 1 : numel(subs)
    
    subject = subs(i);
    subj_str=char(subject);
    
    % Left
    if strcmp(hemi,'left') || strcmp(hemi,'both')
        name = [prefix, '_', subj_str, '_', pattern, '_left.txt'];
        path = fullfile(civet_dir, subj_str, subdir, name);
        thick_files(i,1) = {path};
        to_keep(i) = exist(path,'file');
    end
    
    %Right
    if strcmp(hemi,'right') || strcmp(hemi,'both')
        idx=1;
        if strcmp(hemi,'both'), idx = 2; end;
        name = [prefix, '_', subj_str, '_', pattern, '_right.txt'];
        path = fullfile(civet_dir, subj_str, subdir, name);
        thick_files(i,idx) = {path};
        to_keep(i) = to_keep(i) && exist(path,'file');
    end
    
end

s = sum(~to_keep);
rejected = [];
if s > 0
   fprintf('Warning: %i subjects had no data!\n', s);
   thick_files = thick_files(to_keep,:);
   subjects = subjects(to_keep);
   rejected = subjects(~to_keep);
end

% Read data
Y = SurfStatReadData(thick_files);

end

