
function varargout=pwc_nofig(yn,x1n,x2n,varargin)
%% Partial Wavelet coherence
% Creates a figure of partial cross wavelet power in units of
% normalized variance.
%
% USAGE: [Rsq,period,scale,coi,sig95]=pwc(yn,x1n,x2n[,settings])
%
% 
% Settings: Pad: pad the time series with zeros? 
% .         Dj: Octaves per scale (default: '1/12')
% .         S0: Minimum scale
% .         J1: Total number of scales
% .         Mother: Mother wavelet (default 'morlet')
% .         MaxScale: An easier way of specifying J1
% .         MakeFigure: Make a figure or simply return the output.
% .         BlackandWhite: Create black and white figures
% .         AR1: the ar1 coefficients of the series 
% .              (default='auto' using a naive ar1 estimator. See ar1nv.m)
% .         MonteCarloCount: Number of surrogate data sets in the significance calculation. (default=300)
%
% Settings can also be specified using abbreviations. e.g. ms=MaxScale.
% For detailed help on some parameters type help wavelet.
%
% Example:
%    t=1:200;
%    pwc(sin(t),sin(t.*cos(t*.01)),sin(t.*cos(t*.01)),'ms',16)
%
%

%Please acknowledge the use of this software package in any publications,
%by including text such as:
%
%   The software for the bias-rectified wavelet power spectrum, 
%   partial wavelet coherence and multiple wavelete coherence was 
%   provided by E. K. W. Ng and T. W. Kwok, which is available at:
%   http://www.cityu.edu.hk/gcacic/wavelet. 
%
%
%   (C) Eric K. W. Ng and T. K. Kowk 2012
%
% -------------------------------------------------------------------------
% 
%   Copyright (C) 2012, Eric K. W. Ng and T. K. Kowk 2012
%   This software may be used, copied, or redistributed as long as it is not
%   sold and this copyright notice is reproduced on each copy made.  This
%   routine is provided as is without any express or implied warranties
%   whatsoever.
%
%---------------------------------------------------------------------------



% parse function arguments
pwcargs;

%-----------:::::::::::::--------- ANALYZE ----------::::::::::::------------

[Y,period,scale,coiy] = wavelet(yn(:,2),dt,Args.Pad,Args.Dj,Args.S0,Args.J1,Args.Mother);%#ok
[X1,period,scale,coix1] = wavelet(x1n(:,2),dt,Args.Pad,Args.Dj,Args.S0,Args.J1,Args.Mother);
[X2,period,scale,coix2] = wavelet(x2n(:,2),dt,Args.Pad,Args.Dj,Args.S0,Args.J1,Args.Mother);

sinv=1./(scale');

% smoothing Y, X1 and X2
sY=smoothwavelet(sinv(:,ones(1,nyn)).*(abs(Y).^2),dt,period,Args.Dj,scale);
sX1=smoothwavelet(sinv(:,ones(1,nx1n)).*(abs(X1).^2),dt,period,Args.Dj,scale);
sX2=smoothwavelet(sinv(:,ones(1,nx2n)).*(abs(X2).^2),dt,period,Args.Dj,scale);

% truncate Y, X1 and X2 to common time interval 
dte=dt*.01; 
idx=find((yn(:,1)>=(t(1)-dte))&(yn(:,1)<=(t(end)+dte)));
Y=Y(:,idx);
sY=sY(:,idx);
coiy=coiy(idx);

idx=find((x1n(:,1)>=(t(1))-dte)&(x1n(:,1)<=(t(end)+dte)));
X1=X1(:,idx);
sX1=sX1(:,idx);
coix1=coix1(idx);

idx=find((x2n(:,1)>=(t(1))-dte)&(x2n(:,1)<=(t(end)+dte)));
X2=X2(:,idx);
sX2=sX2(:,idx);
coix2=coix2(idx);

coi=min(min(coiy,coix1),coix2);

% -------- Cross wavelet -------
Wyx1=Y.*conj(X1);
Wyx2=Y.*conj(X2);
Wx2x1=X2.*conj(X1);
% --------Find angle --------
dte=dt*.01; 
idx=find((yn(:,1)>=(t(1)-dte))&(yn(:,1)<=(t(end)+dte)));
yn=yn(idx,:);
y=yn(:,2);

idx=find((x1n(:,1)>=(t(1)-dte))&(x1n(:,1)<=(t(end)+dte)));
x1n=x1n(idx,:);
x1=x1n(:,2);

idx=find((x2n(:,1)>=(t(1)-dte))&(x2n(:,1)<=(t(end)+dte)));
x2n=x2n(idx,:);
x2=x2n(:,2);


% ----------------------- Wavelet coherence ---------------------------------
sWyx1=smoothwavelet(sinv(:,ones(1,n)).*Wyx1,dt,period,Args.Dj,scale);
sWyx2=smoothwavelet(sinv(:,ones(1,n)).*Wyx2,dt,period,Args.Dj,scale);
sWx2x1=smoothwavelet(sinv(:,ones(1,n)).*Wx2x1,dt,period,Args.Dj,scale);

% calculate the R square value
Rsqyx1=abs(sWyx1).^2./(sY.*sX1);
Ryx1=sqrt(Rsqyx1);
Rsqyx2=abs(sWyx2).^2./(sY.*sX2);
Ryx2=sqrt(Rsqyx2);
Rsqx2x1=abs(sWx2x1).^2./(sX2.*sX1);
Rx2x1=sqrt(Rsqx2x1);
Rsq=(abs(Ryx1-Ryx2.*conj(Rx2x1)).^2)./((1-Ryx2.^2).*(1-Rx2x1.^2));

% make figure
%pwcfig;

varargout={Rsq,period,scale,coi,wtcsig};
varargout=varargout(1:nargout);

function [cout,H]=safecontourf(varargin)
vv=sscanf(version,'%i.');
if (version('-release')<14)|(vv(1)<7)
    [cout,H]=contourf(varargin{:});
else
    [cout,H]=contourf('v6',varargin{:});
end

function hcb=safecolorbar(varargin)
vv=sscanf(version,'%i.');

if (version('-release')<14)|(vv(1)<7)
    hcb=colorbar(varargin{:});
else
    hcb=colorbar('v6',varargin{:});
end