function [ h ] = plot_horizontal_sorted ( x_data, params, x_error, sig_axis )

% Plots x_data as a horizontal bar graph, sorted by order
%
% x_data        Data to plot - M x N vector
% params        [Optional] A struct with the following fields:
% y_labels      Labels for y axis - N x 1 cell vector; use []
%               for none
% sort_order    order of sort; one of 'ascend' (default) or 'descend'
% plot_title    Title for plot; [] for none
% limits        Data limits of plot (default is data range)
% plot_size     Window size
% max_bars      Maximum number of bars; default (-1) is everything
% sort_group    The column to sort by
% legend_labels The labels to show in the legend; empty (default) indicates
%               no legend
% plot_type     One of 'grouped' or 'stacked' 
%
% x_error       [Optional] Values defining error bars
%
% Author: Andrew Reid, MNI, 2012

[M,N] = size(x_data);

if nargin < 2
   params = plot_horizontal_sorted_params; 
end

if ~isfield(params,'sort_order') || isempty(params.sort_order)
   params.sort_order = 'ascend'; 
end

if ~isfield(params,'sort_group') || isempty(params.sort_group)
   params.sort_group = 1; 
end

if ~isfield(params,'plot_type') || isempty(params.plot_type)
   params.plot_type = 'grouped'; 
end

% Sort data on first group
x_1 = x_data(:,params.sort_group);
[~, sort_idx] = sort(x_1,1,params.sort_order);
x_data=x_data(sort_idx,:);
if isfield(params,'max_bars') && params.max_bars < M && params.max_bars > 0
   x_data = x_data(1:params.max_bars,:);
   [M,N] = size(x_data);
   sort_idx = sort_idx(1:params.max_bars);
end

% Flip data to plot largest to smallest
idx=M:-1:1;
x_data = x_data(idx,:);
sort_idx = sort_idx(idx);

% Flip column order to plot top-down
idx=N:-1:1;
x_data = x_data(:,idx);

if ~isfield(params,'limits') || isempty(params.limits)
   params.limits = [min(x_data(:)), max(x_data(:))]; 
end

% Plot data
h = figure;
set(h,'Color','w');

g = barh ( x_data, params.plot_type );
xlim(params.limits);
ylim([0, M + 1]);
ax1 = gca;

if exist('x_error','var') && ~isempty(x_error);
    hold on;
    x_error = x_error(sort_idx,:);
    if N > 1
        for col = 1:N
            y = get(get(g(col),'children'),'ydata');
            herrorbar(x_data(:,col),mean(y,1)',zeros(M,1),x_error(:,col), '.k');
        end
    else
        y = get(get(g,'children'),'ydata');
        herrorbar(x_data,mean(y,1)',zeros(M,1),x_error, '.k');
    end
    hold off;
end

if exist('sig_axis','var')
    sig_axis=sig_axis(sort_idx,:);
    ax2 = axes('Parent',h,'Position',get(ax1,'Position'),...
               'YAxisLocation','right',...
               'Color','none',...
               'YColor','k');
    set(ax2, 'YTick', [1:M]);
    set(ax2, 'YTickLabel', sig_axis);
    ylabel(ax2,sig_axis);
end

if isfield(params,'color_map') && ~isempty(params.color_map)
    colormap(params.color_map);
end

if isfield(params,'y_labels') && ~isempty(params.y_labels)
    y_labels = params.y_labels(sort_idx);
    set(ax1, 'YTick', [1:M]);
    set(ax1, 'YTickLabel', y_labels);
end

if isfield(params,'plot_title') && ~isempty(params.plot_title)
    hh = title(params.plot_title);
    set (hh, 'FontSize', 14);
    set (hh, 'FontWeight', 'bold');
end

if isfield(params,'plot_size') && ~isempty(params.plot_size)
   p = get(h,'Position');
   p(1)=100;
   p(2)=100;
   p(3)=params.plot_size(1);
   p(4)=params.plot_size(2);
   set(h,'Position',p);
end

if isfield(params,'legend_labels') && ~isempty(params.legend_labels)
    % Invert
    idx=N:-1:1;
    legend(g(idx),params.legend_labels);
end


