function [ results, models ] = wilson_cowan_parallel_sim ( models, params )

% WILSON_COWAN_PARALLEL_SIM Runs a set of model simulations in parallel,
%                           given parameters:
%
%  General:
%   params.description  - test description of this run
%   params.run_id       - identifier for this run
% 
%  Simulation options
%   params.sim_options  - see wilson_cowan_sim_options
%
%  Wavelet options - morphology
%   params.wav_name     - name of wavelet to use
%   params.wav_scales   - scales (frequency band) to use for transform
%
%  Wavelet coherence options
%   params.wcoh_name    - name of wavelet to use
%   params.wcoh_fs      - signal sampling frequency (hz)
%   params.wcoh_dt      - 1/2 time window for wavelet decomposition (s)
%   params.wcoh_t_step  - time step for computation (samples)
%   params.wcoh_band    - frequency band within which to compute mean coherence
%
%  Morphology options
%   params.morph_params - see activity_growth_thres_params
%
%  Output options
%   params.output_dir   - directory in which to write output
%   params.scratch_dir  - directory for temporary results files
%   params.n_output_files - Number of output files to create
%

% Add paths if necessary
if ~isempty(params.paths)
    for ii = length(params.paths) : 1
       addpath( genpath(params.paths{ii}) );
    end
end

% Empty/create scratch dir
if ~isempty(params.scratch_dir)
    if exist(params.scratch_dir, 'dir')
       fprintf('Removing %s\n', params.scratch_dir);
       rmdir(params.scratch_dir,'s');
    end
    mkdir(params.scratch_dir);
end

prog_file = fullfile(params.log_dir, ... 
                     sprintf('wilson_cowan_sim_progress_%i.log', params.run_id));
if exist(prog_file,'file')
   delete(prog_file); 
end
fid=fopen(prog_file,'w');
fprintf(fid,'Progress for parallel Wilson Cowan simulation: %s\n', params.description);
fprintf(fid,'Trials: %i\n', params.n_trials);
fprintf(fid,'Delays: %i\n', params.n_delays);
fprintf(fid,'Weights: %i\n', params.n_weights);
fclose(fid);

options = params.sim_options;
morph_params = params.morph_params;

N_m = length(models);
results={};
is_output=nargout > 0;
if is_output
   models2 = cell(N_m,1);
   results = cell(N_m,1); 
end

if ~isempty(params.output_file)
    filenames = cell(N_m,1);
    for ii = 1 : N_m
       filenames(ii) = {fullfile(params.scratch_dir, sprintf('temp_%i.mat', ii))}; 
    end
end

elapsed = 0;
t_start = tic;

wcoh_params.w_name = params.wcoh_name;
wcoh_params.dt = params.wcoh_dt;
wcoh_params.fs_o = params.wcoh_fs;
wcoh_params.fs_d = params.wcoh_fs;
wcoh_params.t_step = params.wcoh_t_step;
wcoh_params.scales = params.wav_scales;

parfor i = 1 : N_m
    
    tic;
    fid = fopen(prog_file,'a');
    fprintf(fid,'\nStarting itr %i/%i', i, N_m);
    
    fclose(fid);
    
    % Simulate activity
    model = models(i);
    N_v = length(model.vertices);
    result = wilson_cowan_sim ( model, options );
    fid = fopen(prog_file,'a');
    fprintf(fid,'\n(%i/%i) Simulated activity (%1.2f s)', i, N_m, toc);
    fclose(fid);
    
    % Wavelet coherence of activity
    N_pair = (N_v^2-N_v)/2;
    result.wcoh = zeros(options.t_stop, N_pair);
    itr = 1;
    [ wcoh_all, tt, ff, cwts ] = wcoh_signals( result.E, wcoh_params, result.t );
    c = 1;
    for j = 1 : N_v
        for k = j + 1 : N_v
            % Upsample coherence to original time series
            %x = tt * params.wcoh_fs;
            wcoh = squeeze(wcoh_all(c,:,:));
            %wcoh = interp1(x, wcoh, result.t);
            
            % Fix NaN padding
            idx = find(~isnan(wcoh), 1, 'first');
            if idx > 1, wcoh(1:idx-1)=wcoh(idx); end
            idx = find(~isnan(wcoh), 1, 'last');
            if idx < result.t(end), wcoh(idx+1:end)=wcoh(idx); end
                          
            % Mean coherence across desired frequency band
            bmin = find(ff < params.wcoh_band(2), 1, 'first');
            bmax = find(ff > params.wcoh_band(1), 1, 'last');
            result.wcoh(:,itr) = mean(wcoh(:,bmin:bmax),2);
            itr = itr + 1;
            if j == 1
                result.wcoh_freq = ff;
            end
            c = c + 1;
        end
    end
    fid = fopen(prog_file,'a');
    fprintf(fid,'\n(%i/%i) Computed wavelet coherence (%1.2f s)', i, N_m, toc);
    fclose(fid);
    
    % Wavelet transform activity
    %result.E_wav = cwt_tseries(result.E, params.wav_scales, params.wav_name);
    result.E_wav = cwts;
        % Convert to absolute power
    result.E_wav = sum(abs(result.E_wav),3);
        % Smooth activity across time
    result.E_wav = smooth_tseries(result.t, result.E_wav,100);
    result.E_wav = smooth_tseries(result.t, result.E_wav,100);
        % Apply logistic to transform to [0 1]
    result.E_wav = logistic(result.E_wav,0.05,1);
    result.E_wav = (result.E_wav - 0.5) * 2;
    fid = fopen(prog_file,'a');
    fprintf(fid,'\n(%i/%i) Wavelet transformed activity (%1.2f s)', i, N_m, toc);
    fclose(fid);
    
    % Simulate morphology
    N_v = length(model.vertices);
    morph_init = ones(N_v,1) * morph_params.morph_mean;
    morph_results = activity_growth_sim_gauss (morph_init, result.t, ...
                                                result.E_wav, ...
                                                morph_params);
                                                
    result.morph = morph_results.morph;
    
    % Save to temporary output file here
    if is_output
        models2(i) = {model};
        results(i) = {result};
    end
    if ~isempty(params.output_file)
        parsave(filenames{i}, model, result, params);
    end
    
    fid = fopen(prog_file,'a');
    t = toc;
    elapsed = elapsed + t;
    fprintf(fid,'\n(%i/%i) Simulated morphology (%1.2f s)', i, N_m, t);
    fprintf(fid,'\nDone itr %i/%i (%1.2f s)', i, N_m, toc);
    fclose(fid);
    
end

fid = fopen(prog_file,'a');
t_elapsed = toc(t_start);
str = datestr(datenum(0,0,0,0,0,t_elapsed),'HH:MM:SS');
fprintf(fid,'\nAll done. (Elapsed: %s)', str);
fclose(fid);
clear model; clear result;

if nargout > 0
    models = models2;
end

if ~isempty(params.output_file)
   % Merge temporary files into n files
   merge_files(params, N_m, filenames, prog_file);
   if params.clean_scratch
       rmdir(params.scratch_dir,'s');
   end
end

end

% Save within parfor loop
function parsave( filename, model, result, params )
    save(filename, 'model', 'result', 'params');
end

function merge_files( params, N_m, filenames, prog_file )

N_per = floor(N_m / params.n_output_files);
add_one = N_per ~= N_m / params.n_output_files;
l = 1;
for j = 1 : params.n_output_files
   idx = l : l + N_per - 1;
   if j == params.n_output_files && add_one, idx = [idx N_m]; end
   models = cell(numel(idx),1);
   results = cell(numel(idx),1);
   m = 1;
   for k = idx
       load(filenames{k},'model','result');
       models(m) = {model};
       results(m) = {result};
       m = m + 1;
   end
   outputs = {results, models};
   file = fullfile(params.output_dir, sprintf('%s_part%i.mat', params.output_file, j));
   save(file, 'outputs');
   fid = fopen(prog_file,'a');
   if j == 1, fprintf(fid,'\n'); end;
   fprintf(fid,'\nResults written to ''%s''', file);
   fclose(fid);
   l = idx(end) + 1;
end
   
end