function [ result, AA ] = wilson_cowan_sim( model, options )
%WILSON_COWAN_SIM Generates a Wilson-Cowan simulation result
%   
% Options:
% t_stop    - The time at which to stop the stimulation
% dt        - The time step
% 

% Global parameters
N_t = length(1:options.dt:options.t_stop);  % Total number of time points
N_v = length(model.vertices);               % Total number of vertices in model

% Global variables
AA = zeros(N_v, N_t);                       % The activity for each vertex 
                                            % over time 

noise = zeros(N_v, N_t);                    % Gaussian white noise per vertex
for j = 1 : N_v
   vertex = model.vertices(j);
   noise(j, :) = wgn(1,N_t,1) * vertex.epsilon;
end
                                            
% Get matrices
[ weights, delays ] = wilson_cowan_matrices( model );

% Get initial [E,I] vectors, by concatenating values from all vertices
y_init = zeros(N_v * 3, 1);

% Run ODE solver
ode_options= odeset('MaxStep',1);
%ode_options= odeset('RelTol',1e-3,'AbsTol',1e-3);
[ t_vals, y_vals ] = ode45( @solve, 1:options.dt:options.t_stop, y_init, ode_options );

% Compile result
result.t = t_vals;
result.E = zeros(N_t,N_v);
result.E = zeros(N_t,N_v);
for j = 1 : N_v
    jj = (j-1)*3 + 1;
    result.E(:,j) = y_vals(:,jj);
    result.I(:,j) = y_vals(:,jj+1);
    result.P(:,j) = y_vals(:,jj+2);
end

% ODE solver 
    function dydt = solve ( t, y )
        tt = round (t / options.dt);        % Index of this time point
        dydt = zeros( length(y), 1 );
        
        % Update activity matrix for the current time point
        if tt > t / options.dt              % Bad; fix this to interpolate
            ys = y(1:3:end);
            AA(:,tt) = ys;
        end
        
        % Solve each vertex for this time point
        for i = 1 : N_v
            vertex = model.vertices(i);
            ii = (i-1) * 3 + 1;
            Et = y(ii);
            It = y(ii+1);
            P = 0;
             
            % Sum of input from connected vertices, accounting for delays
            if N_v > 1
                tf = repmat(t,N_v,1) - delays(:,i);    % Activity of connected vertices following
                                                        % delay
                idx = find(tf > 0);                     % Delay must be shorter than elapsed time
                if ~isempty(idx)
                    tfi = round(tf(idx)/options.dt);        % Index of this time point
                    if tfi > 0
                        w_i = weights(idx,i); %repmat(i,length(idx),1));
%                         if sum(w_i) > 0 && vertex.afferent_weight >= 0
%                             ws = vertex.afferent_weight / sum(w_i);
%                             w_i = w_i * ws;
%                         end
                        A1 = zeros(length(tfi),1);
                        for k = 1 : length(tfi)
                           A1(k) = AA(idx(k),tfi(k)); 
                        end
                        if sum(A1) > 0
                           a = 0; 
                        end
                        P = sum(A1.*w_i);                 % Weighted sum of signals converging on this vertex
                        %P = input_sigmoid ( P );          % Input transformed by sigmoid on [0,1]
                    end
                end
            end
            
            % Add external signal(s) if applicable
            if ~isempty(model.params.input_signal)
                P_ext = model.params.input_signal(tt,:).*model.params.input_gain;
                P = P + sum(P_ext.*vertex.external_weight);
            end
            
            P = input_sigmoid ( P );          % Input transformed by sigmoid on [0,1]
            
            % Add noise 
            P = P + noise(i,tt);
            
            % Differential equations
            EE = vertex.p_ee*Et - vertex.p_ei*It - vertex.theta_e + P;
            E = -Et + wc_fcn(vertex.a_e * EE, vertex);
            E = E / vertex.tau_e;
            II = vertex.p_ie*Et - vertex.p_ii*It - vertex.theta_i;
            I = -It + wc_fcn(vertex.a_i * II, vertex);
            I = I / vertex.tau_i;
            dydt(ii:ii+2) = [ E; I; P ];
            
            
        end
        
    end

    % Sigmoidal transfer function for differential equations
    function v = wc_fcn( v, vertex )
        fun = vertex.function;
        params = vertex.function_params;
        if strcmp(fun, 'tanh')
            v = tanh(v);
        elseif strcmp(fun, 'logistic')
            alpha = 1;
            if ~isempty(params), alpha = params(1); end;
            v = 1 / ( 1 + exp(-alpha * v) );
        elseif strcmp(fun, 'naka-rushdon')
            if isempty(params), params = [1 1 1]; end;
            u = params(1); N = params(2); delta = params(3);
            v = (u * v ^ N) / ( delta ^ N + v ^ N);
        end
    end

    function v = input_sigmoid( v )
         v=1./( 1 + 100 * exp(-10 * v) );
    end

end

