%% Wilson-Cowan model runs
%
% General script for generating WC models across weighting parameters and
% random input pulse sequences.
%
% Procedure: 
%
% 1. Set-up and run initiation (choose parallel cluster if relevant)
% 2. Choose hypothesis & run that section; this will generate a set of
%    models ranging over params.n_weights weights and params.n_trials 
%    random pulse sequence inputs
% 3. Run simulation on these models. Choose serial, parallel local, or
%    parallel remote options.
% 4. Run postprocessing and statistical analyses
% 5. Set-up and show the desired plots
% 6. Output plots and results
%

%% 1. Initiation

%% 1.1. Initiate model parameters

params = wilson_cowan_run_params;
%lib_dir = params.lib_dir;

%% 1.2. Set up parallel processing

clear par_params;

%% 1.2.1. Local

% Local host
par_params.cluster = parcluster('local');           % Cluster to use
max_proc = par_params.cluster.NumWorkers - 1;       % Maximum processors available
par_params.n_proc = min(max_proc, params.n_trials * params.n_weights);  % Number of processors to use
params.paths = {};
params.n_output_files = 1;

%% 1.2.2. Remote cluster (Guillimin)

% Guillimin

par_params.cluster = parcluster('Guillimin');       % Cluster to use
max_proc = 24;
%max_proc = par_params.cluster.NumWorkers - 1;       % Maximum processors available
par_params.n_proc = min(max_proc, params.n_trials * params.n_weights); % Number of processors to use
par_params.n_nodes = ceil(par_params.n_proc / 12);  % Set number of node to use
par_params.ppn = min(par_params.n_proc, 12);        % Set processors per node (12 max)
par_params.wall_time = '24:00:00';                  % Set wall time

% Paths to add on remote cluster
params.paths = {'~/matlab/lib_areid'};
params.output_file = '';        % Set this for models with large outputs; 
                                % output variables will be saved as a .mat 
                                % file, rather than being fetched
params.output_dir = '/sb/project/eim-670-aa/areid/wc_models';
params.log_dir = '/sb/project/eim-670-aa/areid/wc_models/logs';
params.lib_dir= '/home/areid/matlab/lib_areid';
params.n_output_files = 1;
params.scratch_dir = '/gs/scratch/areid/wc_models'; % '/sb/scratch/areid/wc_models';    

%% 1.2.3. Local (FZJ)

% Local host
par_params.cluster = parcluster('local');           % Cluster to use
max_proc = par_params.cluster.NumWorkers - 1;       % Maximum processors available
par_params.n_proc = min(max_proc, params.n_trials * params.n_weights);  % Number of processors to use
params.paths = {};
%params.n_output_files = 1;

% Paths to add on remote cluster
params.output_file = '';        % Set this for models with large outputs; 
                                % output variables will be saved as a .mat 
                                % file, rather than being fetched
params.output_dir = '/data/Eickhoff4/Andrew/results/scov_model';
params.log_dir = '/data/Eickhoff4/Andrew/results/scov_model/logs';
params.lib_dir= '/data/Eickhoff4/Andrew/matlab/lib_areid';
%params.n_output_files = 1;
params.scratch_dir = '/data/Eickhoff4/Andrew/scratch'; 


%% 2. Model generation

%% 2.0. One-unit models
%

N_units = 1;

%% 2.0.1. Varying alpha over constant beta

params.run_id = 0;
params.description = '1.0.1. Varying alpha over constant beta';
alphas = 0:1e-5:2e-4;
n_alphas = length(alphas);
% Set this if you want results to be stored in a .mat file
params.output_file = sprintf('wc_results_1_0_1_alphas%i_id%i', ...
                             n_alphas, params.run_id);

models = generate_wc_models_1_0_1(params, alphas');
job = batch('wilson_cowan_parallel_sim', 2, {models, params}, ...
            'matlabpool', par_params.n_proc)
%% 2.0.2. Varying beta over constant alpha

params.run_id = 0;
params.description = '1.0.2. Varying beta over constant alpha';
betas = 0:.1:2;
n_betas = length(betas);
% Set this if you want results to be stored in a .mat file
params.output_file = sprintf('wc_results_1_0_2_alphas%i_id%i', ...
                             n_betas, params.run_id);

models = generate_wc_models_1_0_2(params, betas');

%% 2.1. Two-unit models
%

N_units = 2;

%% 2.1.1. Unidirectional pair: V1 -> V2

%% 2.1.1.1. Hypothesis: Unidirectional connectivity strength is proportional 
%                       to morphological correlation. Vary weights from 0 to 1.

params.run_id = 2;
params.description = '1.1.1. Unidirectional two-vertex model';
% Set this if you want results to be stored in a .mat file
params.output_file = sprintf('wc_results_1_1_1_n%i_w%i_id%i', ...
                             params.n_trials, params.n_weights, params.run_id);
params.n_delays = 1;
% delays = (0:100/(params.n_delays-1):100)';
% delays = [ones(params.n_delays,1), delays];
weight_max = 1.5;
weights = (0:weight_max/(params.n_weights-1):weight_max)';
models = generate_wc_models_1_1_1(params, weights);

%% 2.1.2. Bidirectional pair: V1 -> V2, V2 -> V1

%% 2.1.2.1. Hypothesis: Bidirectional connectivity increases morphological
%                     correlation. Vary V1 -> V2 while keeping V2 -> V1
%                     constant.

params.run_id = 2;
params.description = '2.1.2.1. Bidirectional two-vertex model';
% Set this if you want results to be stored in a .mat file
params.output_file = sprintf('wc_results_2_1_2_1_n%i_w%i_id%i.mat', ...
                             params.n_trials, params.n_weights, params.run_id);

weights = (0:1/(params.n_weights-1):1)';
weights = [ones(params.n_weights,1) * 0.8, weights];
delays = (0:100/(params.n_delays-1):100)';
delays = [delays, delays];
models = generate_wc_models_1_1_2(params, weights, delays);
weight_labels = {'V1->V2','V2->V1'};

%% 2.2. Three-unit models

N_units = 3;

%% 2.2.1. Sequential unidirectional connectivity: V1 -> V2 -> V3

%% 2.2.1.1. Hypothesis: Indirect connectivity leads to morphological 
%                     correlation (V1 & V3)

%% 2.2.1.2. Hypothesis: Indirect morphological correlations are weaker than 
%                     direct correlations

params.run_id = 4;
params.description = '2.2.1.2. Sequential three-vertex model';
% Set this if you want results to be stored in a .mat file
params.output_file = sprintf('wc_results_2_2_1_2_n%i_w%i_id%i', ...
                             params.n_trials, params.n_weights, params.run_id);
params.n_delays = 1;
delays = [0 0];
weight_max = 1.5;
weights = (0:weight_max/(params.n_weights-1):weight_max)';
weights = [ones(params.n_weights,1) * 0.5, weights];

params.pcoh_triplets=[[1,2,3];[1,3,2];[2,3,1]];

models = generate_wc_models_2_1_1(params, weights, delays);

%% 2.2.2. Common afferent: V1 -> V3 and V2 -> V3
%

%% 2.2.2.1. Hypothesis: V3 will be morph-correlated with both V1 and V2
%

params.run_id = 5;
params.description = '2.2.2.1. Common afferent three-vertex model';
% Set this if you want results to be stored in a .mat file
params.output_file = sprintf('wc_results_2_2_2_1_n%i_w%i_id%i', ...
                             params.n_trials, params.n_weights, params.run_id);
params.n_delays = 1;
delays = [0 0];
weight_max = 1.5;
weights = (0:weight_max/(params.n_weights-1):weight_max)';
weights = [ones(params.n_weights,1) * 0.5, weights];

models = generate_wc_models_2_2_1(params, weights, delays);

%% 2.2.2.2. Hypothesis: Morph-correlation of V1 & V3 is inversely proportional to 
%                     the weight of V2 -> V3. Keep V1 -> V3 constant while
%                     varying V2 -> V3.

params.run_id = 6;
params.description = '2.2.2.2. Common afferent three-vertex model';
% Set this if you want results to be stored in a .mat file
params.output_file = sprintf('wc_results_2_2_2_2_n%i_w%i_id%i', ...
                             params.n_trials, params.n_weights, params.run_id);
params.n_delays = 1;
delays = [0 0];
weight_max = 1.5;
weights = (0:weight_max/(params.n_weights-1):weight_max)';
weights = [ones(params.n_weights,1) * 0.5, weights];

models = generate_wc_models_2_2_1(params, weights, delays);

%% 2.2.3. Common efferent: V1 -> V2 and V1 -> V3

%% 2.2.3.1. Hypothesis: V2 and V3 will be correlated

%% 2.2.3.2. Hypothesis: Morh-correlation of V2 & V3 is proportional to the sum 
%                     (product?) of their connection strengths V1 -> V2
%                     and V1 -> V3. Keep V1 -> V2 constant while varying 
%                     V1 -> V3. Additionally, randomly vary both.

params.run_id = 8;
params.description = '2.2.3.2. Common efferent three-vertex model';
% Set this if you want results to be stored in a .mat file
params.output_file = sprintf('wc_results_2_2_3_2_n%i_w%i_id%i', ...
                             params.n_trials, params.n_weights, params.run_id);
params.n_delays = 1;
delays = [0 0];
weight_max = 1.5;
weights = (0:weight_max/(params.n_weights-1):weight_max)';
weights = [ones(params.n_weights,1) * 0.5, weights];

models = generate_wc_models_2_2_3(params, weights, delays);


%% 2.2.4. Common afferent with one bidirectional connection
%       i.e., V1 - > V2, V2 -> V1, V1 -> V3
%

%% 2.2.4.1. Hypothesis: Morph-correlation of bidirectional connection is 
%                     stronger than for unidirectional. Use identical
%                     weights.

%% 2.3. Four-unit models


%% 2.4. Five-unit models

N_units = 5;

%% 2.4.1. Common afferent, common efferent

%% 2.4.1.1. Hypothesis: Morph-correlation will vary proportional to edge weights,
%                       inlcuding indirect sequential correlations. Hub node
%                       will have strongest morph-correlations
%                       Configuration: Input1 -> V1, Input2 -> V2
%                                      V1,V2 -> V3
%                                      V3 -> V4,V5
%

params.run_id = 9;
params.description = '2.4.1.1. Common efferent, afferent five-vertex model';
% Set this if you want results to be stored in a .mat file
params.output_file = sprintf('wc_results_2_4_1_1b_n%i_w%i_id%i', ...
                             params.n_trials, params.n_weights, params.run_id);
params.n_delays = 1;
delays = [0 0 0 0];
weight_max = 1.5;
static_wt = ones(params.n_weights,1) * 0.5;

% Run four times, varying weight for different edge in each run
edge_to_vary = 3;
weights = repmat(static_wt, 1, 4);
weights(:,edge_to_vary) = (0:weight_max/(params.n_weights-1):weight_max)';

models = generate_wc_models_2_4_1_1(params, weights,delays);


%% 2.5. Random models with symmetry

N_units = 7;

%% 2.5.1. Randomly generate feedforward networks:


%% 2.5.1.1. No external input

% Hypothesis: Morph-correlation will vary proportional to pairwise symmetry,
%                       a la Nicosa et al., 2013
%

params.run_id = 10;
params.description = '2.5.1.1. Random symmetry (no external input)';

% Set this if you want results to be stored in a .mat file
params.output_file = sprintf('wc_results_2_5_1_1_n%i_w%i_id%i', ...
                             params.n_trials, params.n_weights, params.run_id);
params.n_delays = 1;
params.n_probs = 5;
delays = zeros(N_units,1);
weight_min = 0.1;
weight_max = 1;

% Varying weight for all edges
weights = (weight_min:(weight_max-weight_min)/(params.n_weights-1):weight_max)';

% Rewiring probabilities
probs = [0 : 1 / (params.n_probs-1) : 1];

models = generate_wc_models_2_5_1_1(params, weights, delays, probs);

%% 2.5.1.2. External input to V1

% Hypothesis: Morph-correlation will vary proportional to pairwise symmetry,
%                       a la Nicosa et al., 2013
%

% Hypothesis: Morph-correlation will vary proportional to pairwise symmetry,
%                       a la Nicosa et al., 2013
%

params.run_id = 11;
params.description = '2.5.1.2. Random symmetry (external input to V1)';

% Set this if you want results to be stored in a .mat file
params.output_file = sprintf('wc_results_2_5_1_1_n%i_w%i_id%i', ...
                             params.n_trials, params.n_weights, params.run_id);
params.n_delays = 1;
params.n_probs = 1;
delays = zeros(N_units,1);
weight_min = 0.1;
weight_max = 1;

% Varying weight for all edges
weights = (weight_min:(weight_max-weight_min)/(params.n_weights-1):weight_max)';

% Rewiring probabilities
probs = [0 : 1 / (params.n_probs-1) : 1];

models = generate_wc_models_2_5_1_2(params, weights, delays, probs);


%% 3. Run simulations on all models
%
% Note that, in Linux, you can follow the log file at
% {params.log_dir}/{params.output_file}.log with the command:
%
% tail -f /path/to/logs/log_file.log
%

%% 3.1. As serial job

[results, models] = wilson_cowan_parallel_sim( models, params );

results = cell2mat(results)';
models = cell2mat(models);

M = results;
clear results;
l = 1;
for i = 1 : params.n_trials
    for j = 1 : params.n_delays
        for k = 1 : params.n_weights
           results(i,j,k) = M(l); 
           l = l + 1;
        end
    end 
end
M = models;
clear models;
l = 1;
for i = 1 : params.n_trials
    for j = 1 : params.n_delays
        for k = 1 : params.n_weights
           models(i,j,k) = M(l); 
           l = l + 1;
        end
    end 
end

%% 3.2. As parallel job - local
%
% Runs as a parallel job, using the local multiprocessor machine
%

job = batch('wilson_cowan_parallel_sim', 2, {models, params}, ...
            'matlabpool', par_params.n_proc);

wait( job );                    % Wait for this job to finish
outputs = fetchOutputs(job);    % Fetch the results

M = outputs{1};
clear results;
l = 1;
for i = 1 : params.n_trials
    for p = 1 : params.n_probs
        for j = 1 : params.n_delays
            for k = 1 : params.n_weights
               a = M{l};
               if ~isempty(a)
                    results(i,p,j,k) = M{l}; 
               end
               l = l + 1;
            end
        end
    end 
end
M = outputs{2};
clear models;
l = 1;
for i = 1 : params.n_trials
    for p = 1 : params.n_probs
        for j = 1 : params.n_delays
            for k = 1 : params.n_weights
               a = M{l};
               if ~isempty(a)
                    models(i,p,j,k) = M{l}; 
               end
               l = l + 1;
            end
        end 
    end
end

%% 3.3. As parallel job - Guillimin
%params.output_file
% Note 1: you can skip the "wait" command to avoid blocking while jobs sit on
% the remote queue (just highlight the first two lines and evaluate them). 
% This requires you monitor the job and run the subsequent commands once it 
% has finished. You can do this with Matlab's "Parallel > Monitor Jobs" dialog, 
% or by viewing the "job" variable, or by monitoring the remote log file 
% (located in params.output_dir) called "wilson_cowan_sim_progress_<run_id>.log"
%
% Note 2: if you start a job and then exit Matlab, it can be retrieved by the
% command: 
%
% > job = par_params.cluster.findJob('ID',<job_id>)
%
% The ID can be determined as job.ID; or by using the 
% "Parallel > Monitor Jobs" dialog
%

% set_moab_file(fullfile(lib_dir, 'models', 'wilson_cowan'), ...
%               par_params.n_nodes, par_params.ppn, par_params.wall_time);

set_moab_file(fullfile(params.lib_dir, 'models'), ...
              par_params.n_nodes, par_params.ppn, par_params.wall_time);

if isempty(params.output_file)
    % Run this if you want to retrieve outputs directly from the job, or
    % you haven't specified an output file (params.output_file)
    job = batch(par_params.cluster,'wilson_cowan_parallel_sim', 2, {models, params}, ...
              'matlabpool', par_params.n_proc); %, 'AdditionalPaths', par_params.lib_paths);
    wait( job );                    % Wait for this job to finish
    outputs = fetchOutputs(job);    % Fetch the results
    
else
    % Run this, then download the .mat file (params.output_file) from the
    % remote site & open it
    job = batch(par_params.cluster,'wilson_cowan_parallel_sim', 0, {models, params}, ...
              'matlabpool', par_params.n_proc); %, 'AdditionalPaths', par_params.lib_paths);

end

M = outputs{1};
clear results;
l = 1;
for i = 1 : params.n_trials
    for p = 1 : params.n_probs
        for j = 1 : params.n_delays
            for k = 1 : params.n_weights
               results(i,p,j,k) = M{l}; % Change
               l = l + 1;
            end
        end 
    end
end
M = outputs{2};
clear models;
l = 1;
for i = 1 : params.n_trials
    for p = 1 : params.n_probs
        for j = 1 : params.n_delays
            for k = 1 : params.n_weights
               models(i,p,j,k) = M{l}; % Change
               l = l + 1;
            end
        end 
    end
end


%% 4. Postprocessing & Analyses
%
% Perform statistical analyses on the results.
%

%% 4.1. Morphology correlations across weights

morph_corr_n_rnd = 1000;                    % Random samples from each morphometry

morph_final = zeros(params.n_trials,params.n_delays,params.n_weights,params.n_probs,morph_corr_n_rnd,N_units);
idx = randsample(params.sim_options.t_stop,morph_corr_n_rnd);

for i = 1 : params.n_trials
    %st_idx = (i-1) * morph_corr_n_rnd + 1;
    for p = 1 : params.n_probs
        for j = 1 : params.n_delays
            for k = 1 : params.n_weights
                result = results(i,p,j,k);
                %morph_final(st_idx:st_idx+morph_corr_n_rnd-1,j,k,:) = result.morph(idx,:);
                morph_final(i,p,j,k,:,:) = result.morph(idx,:);
            end
        end
    end
end

%N_top = N_units * (N_units-1) / 2;
morph_corr = zeros(morph_corr_n_rnd,params.n_delays,params.n_weights,params.n_probs,N_units,N_units);
%for i = 1 : params.n_trials
for p = 1 : params.n_probs
    for j = 1 : params.n_delays
        for k = 1 : params.n_weights
            for l = 1 : morph_corr_n_rnd
                mm = squeeze(morph_final(:,p,j,k,l,:));
                morph_corr(l,p,j,k,:,:)= corrcoef(mm);
            end
            %morph_corr(i,j,k,:,:) = corrcoef(m);
        end
    end
end
%end

morph_corr(isnan(morph_corr))=0;

%% 4.2. Mean wavelet coherence across weights

% Get mean for each trial, for each weight and delay
N_comp = N_units * (N_units-1) / 2;
wcoh_means = zeros(params.n_trials, params.n_delays, params.n_weights, params.n_probs, N_comp);
for i = 1 :  params.n_trials 
    for p = 1 : params.n_probs
        for j = 1 : params.n_delays
           for k = 1 : params.n_weights
               result = results(i,p,j,k);
               for u = 1 : N_comp
                    T = result.wcoh(:,u);
                    T = T(~isnan(T));
                    wcoh_means(i,p,j,k,u) = mean(T);
               end
           end
        end
    end
end

% Get mean, stdev across trials
wcoh_mean = zeros(params.n_probs, params.n_delays, params.n_weights, N_comp);
wcoh_std = zeros(params.n_probs, params.n_delays, params.n_weights, N_comp);

for p = 1 : params.n_probs
    for j = 1 : params.n_delays
        for k = 1 : params.n_weights
            for u = 1 : N_comp
                wcoh_mean(p,j,k,u) = mean(wcoh_means(:,p,j,k,u));
                wcoh_std(p,j,k,u) = std(wcoh_means(:,p,j,k,u));
            end
        end
    end
end

%% 4.3.a. Compute partial coherence (Parallel)

if ~isempty(params.pcoh_triplets)
    job = batch(par_params.cluster,'compute_partial_coherence', 1, {params, results, params.pcoh_triplets}, ...
                  'matlabpool', par_params.n_proc);

    wait( job );                    % Wait for this job to finish
    outputs = fetchOutputs(job);
    results = outputs{:};
end

%% 4.3.b. Compute partial coherence (Serial)

if ~isempty(params.pcoh_triplets)
    results = compute_partial_coherence(params, results, params.pcoh_triplets);
end

%% 4.4. Save summary


%% 5. Plots

clear plot_params;

plot_params.show_legends = true;       % Show legends?
plot_params.line_width = 2;             % Line widths for plots
plot_params.show_axis_labels = true;   % Show axis labels?
plot_params.show_titles = true;        % Show titles?
plot_params.line_colours=[];

% Specify which trials to plot
plot_params.idx_trial = [ ceil(rand * params.n_trials) ];
N_idx_trial = length (plot_params.idx_trial);

% Specify which delays to plot
plot_params.idx_delay = [ 1 ];% 2 3 ]; % 8 10 ];
N_idx_delay = length(plot_params.idx_delay);

% Specify which weights to plot
plot_params.idx_weight = [ 1 2 3 ];

% Specify which probabilities to plot
plot_params.idx_prob = [ 1 ];

%plot_params.idx_weight = [ 1 round(params.n_weights / 2) params.n_weights ];
N_idx_weight = length (plot_params.idx_weight);

% Specify which vertices to plot
plot_params.idx_vertex = 1 : N_units;
N_idx_vertex = length(plot_params.idx_vertex);

% Specify which edges to plot
% plot_params.idx_edge = 1 : size(weights,2);
% N_idx_edge = length(plot_params.idx_edge);

axis_idx = 1;               % Indices of edge(s) whose weight to use as axis
plot_edge_idx = [1,2;1,5;1,4;1,7;6,7;4,5;2,3;2,4];          % Indices of edge(s) to plot 
N_plot_edge_idx = size(plot_edge_idx,1);

model = models(1,1);
vertices = model.vertices;

c_names = {};
l_names = {};
for k = 1 : N_plot_edge_idx
       c_names = [c_names, {sprintf('%s->%s', vertices(plot_edge_idx(k,1)).label, ...
                                                 vertices(plot_edge_idx(k,2)).label)}];
       l_names = [l_names, {sprintf('%s/%s', vertices(plot_edge_idx(k,1)).label, ...
                                              vertices(plot_edge_idx(k,2)).label)}];
end

% Specify plot sizes
plot_params.ts_plot_size_input = [1700 150];
plot_params.ts_plot_size_multi = [1700 300];

% Time series
ts = results(1,1,1).t;

% Labels
plot_params.idx_edge = 1 : size(weights,2);
N_idx_edge = length(plot_params.idx_edge);
model = models(1, 1);
edges = model.edges;
vertices = model.vertices;
weight_labels = cell(N_idx_edge,1);
for i = 1 : N_idx_edge
   edge = edges(i);
   weight_labels(i) = {sprintf('%s->%s', vertices(edge.from).label, vertices(edge.to).label)};
end

% Graphs
plot_params.graph_show_arrows=true;

model = models(plot_params.idx_trial,1,1,plot_params.idx_prob);

legend_text = cell(N_idx_vertex,1);
for k = 1 : N_idx_vertex
    vertex = vertices(plot_params.idx_vertex(k));
    legend_text(k) = {vertex.label};
end

%% 5.0. Set model & results to plot
models_idx = models(plot_params.idx_trial, :, :, :);
model = models(1,1,1,1);
results_idx = results(plot_params.idx_trial, :, :, :);
clear outputs;

%% 5.1. Plot input pulse sequences
%
% Plots the input sequence(s) for the models corresponding to trial
% plot_params.idx_trial, as filled boxes.
%

h_plot_inputs = zeros(N_idx_trial, 1);
for i = 1 : N_idx_trial
    idx = plot_params.idx_trial(i);
    model = models_idx(i,1,1,plot_params.idx_prob);
    inputs = model.params.input_signal;
    %inputs(:,2)=inputs(:,2)*0.6;
    ts = results(1,1,1).t;

    if isempty(plot_params.line_colours)
        colours = [0,0,1;0,0.5,0];
    else
        colours = plot_params.line_colours;
    end
    h = figure;
    set(h,'Color','w');
    for b = 1 : size(inputs,2);
        P = squeeze(inputs(:,b));
        hh = area(ts, P);
        set(hh(1),'FaceColor',colours(b,:));
        hold on;
    end
    
    set(gca, 'XTick', get(gca, 'XTick'), 'XTickLabel', sprintf('%i|', get(gca, 'XTick')));
    
    ylim([0 1.2]);
    if plot_params.show_titles
        hh = title (sprintf('Input sequence(s) for trial #%i', idx));
        set (hh, 'FontSize', 14);
        set (hh, 'FontWeight', 'bold');
    end
    hold off;

    resize_window( h, plot_params.ts_plot_size_input, [100 850] );
    h_plot_inputs(i) = h;
end

%% 5.2. Plot raw activity as time series

model = models_idx(1,plot_params.idx_prob,1,1);
inputs = model.params.input_signal;
ts = results(1,1,1,1).t;

h_plot_activity = zeros(N_idx_trial,N_idx_delay);
for i = 1 : N_idx_trial
    idx_i = plot_params.idx_trial(i);
    
    for d = 1 : N_idx_delay
        idx_d = plot_params.idx_delay(d);
        h = figure;
        set (h, 'Color', 'w');
        for w = 1 : N_idx_weight
            idx_w = plot_params.idx_weight(w);
            %result = results(idx_i,idx_d,idx_w);
            result = results_idx(i,plot_params.idx_prob,idx_d,idx_w);
            subplot(N_idx_weight,1,w); 
            if isempty(plot_params.line_colours)
                hh=plot(ts, result.E(:,plot_params.idx_vertex));
            else
                for a = 1 : N_units
                    hh = plot(ts, result.E(:,plot_params.idx_vertex(a)), 'Color', plot_params.line_colours(a,:));
                    hold on;
                end
                hold off;
            end
            set(gca, 'XTick', get(gca, 'XTick'), 'XTickLabel', sprintf('%i|', get(gca, 'XTick')));
    
            txt = '';
            for k = 1 : length(weight_labels)
                txt = [txt sprintf('w(%s)=%1.2f ', weight_labels{k}, weights(idx_w,k))];
            end
            if plot_params.show_axis_labels
                xlabel(txt);
            end
            ylim([0 1]);
        end
        if plot_params.show_titles
            if params.n_delays > 1
                hh = suptitle(sprintf('Simulated activity for trial #%i (delay1=%1.2f)', idx_i, delays(idx_d,1)));
            else
                hh = suptitle(sprintf('Simulated activity for trial #%i', idx_i));
            end
            set(hh, 'FontSize', 14);
            set(hh, 'FontWeight', 'bold');
        end
        if plot_params.show_legends
            legend(legend_text);
        end
        
        resize_window( h, plot_params.ts_plot_size_multi, [100 500] );
        h_plot_activity(i,d) = h;
    end
end

%% 5.3. Plot wavelet-transformed activity as time series

%model = models(plot_params.idx_trial, 1);
model = models_idx(1,plot_params.idx_prob,1,1);
edges = model.edges;
vertices = model.vertices;
%weight_labels = cell(N_idx_edge,1);

h_plot_wavelet = zeros(N_idx_trial,N_idx_delay);
for i = 1 : N_idx_trial
    idx_i = plot_params.idx_trial(i);
    
    for d = 1 : N_idx_delay
        idx_d = plot_params.idx_delay(d);
        h = figure;
        set (h, 'Color', 'w');
        for w = 1 : N_idx_weight
            idx_w = plot_params.idx_weight(w);
            %result = results(idx_i,idx_d,idx_w);
            result = results_idx(i,idx_d,idx_w);
            subplot(N_idx_weight,1,w);
            if isempty(plot_params.line_colours)
                hh = plot(ts, result.E_wav(:,plot_params.idx_vertex));
                set(hh,'LineWidth',plot_params.line_width);
            else
                for a = 1 : N_units
                    hh = plot(ts, result.E_wav(:,plot_params.idx_vertex(a)), 'Color', plot_params.line_colours(a,:));
                    set(hh,'LineWidth',plot_params.line_width);
                    hold on;
                end
                hold off;
            end
            set(gca, 'XTick', get(gca, 'XTick'), 'XTickLabel', sprintf('%i|', get(gca, 'XTick')));
    
            txt = '';
            for k = 1 : length(weight_labels)
                txt = [txt sprintf('w(%s)=%1.2f ', weight_labels{k}, weights(idx_w,k))];
            end
            if plot_params.show_axis_labels
                xlabel(txt);
            end
            ylim([0 1]);
        end
        if plot_params.show_titles
            if params.n_delays > 1
                hh = suptitle(sprintf('Wavelet-transformed activity for trial #%i (delay1=%1.2f)', idx_i, delays(idx_d)));
            else
                 hh = suptitle(sprintf('Wavelet-transformed activity for trial #%i', idx_i));
            end
            set(hh, 'FontSize', 14);
            set(hh, 'FontWeight', 'bold');
        end
        if plot_params.show_legends
            legend(legend_text);
        end

        resize_window( h, plot_params.ts_plot_size_multi, [100 500] );
        h_plot_wavelet(i,d) = h;
    end
end

%% 5.4. Plot morphology as time series

%model = models(plot_params.idx_trial, 1);
model = models_idx(1,plot_params.idx_prob,1,1);
edges = model.edges;
vertices = model.vertices;
weight_labels = cell(N_idx_edge,1);
for i = 1 : N_idx_edge
   edge = edges(i);
   weight_labels(i) = {sprintf('%s->%s', vertices(edge.from).label, vertices(edge.to).label)};
end

h_plot_morph = zeros(N_idx_trial,N_idx_delay);
for i = 1 : N_idx_trial
    idx_i = plot_params.idx_trial(i);
    
    for d = 1 : N_idx_delay
        idx_d = plot_params.idx_delay(d);
        h = figure;
        set (h, 'Color', 'w');
        for w = 1 : N_idx_weight
            idx_w = plot_params.idx_weight(w);
            %result = results(idx_i,idx_d,idx_w);
            result = results(i,idx_d,idx_w);
            subplot(N_idx_weight,1,w); 
            
            if isempty(plot_params.line_colours)
                hh = plot(ts, result.morph(:,plot_params.idx_vertex));
                set(hh,'LineWidth',plot_params.line_width);
            else
                for a = 1 : N_units
                    hh = plot(ts, result.morph(:,plot_params.idx_vertex(a)), 'Color', plot_params.line_colours(a,:));
                    set(hh,'LineWidth',plot_params.line_width);
                    hold on;
                end
                hold off;
            end
            set(gca, 'XTick', get(gca, 'XTick'), 'XTickLabel', sprintf('%i|', get(gca, 'XTick')));
    
            ylim([params.morph_params.morph_mean - .1 * params.morph_params.morph_std ...
                  params.morph_params.morph_mean + .1 * params.morph_params.morph_std]);
            txt = '';
            for k = 1 : length(weight_labels)
                txt = [txt sprintf('w(%s)=%1.2f ', weight_labels{k}, weights(idx_w,k))];
            end
            if plot_params.show_axis_labels
                xlabel(txt);
            end
            %ylim([0 1]);
        end
        if plot_params.show_titles
            if params.n_delays > 1
                hh = suptitle(sprintf('Morphology for trial #%i (delay1=%1.2f)', idx_i, delays(idx_d)));
            else
                hh = suptitle(sprintf('Morphology for trial #%i', idx_i));
            end
            set(hh, 'FontSize', 14);
            set(hh, 'FontWeight', 'bold');
        end
        
        if plot_params.show_legends
            legend(legend_text);
        end

        resize_window( h, plot_params.ts_plot_size_multi, [100 500] );
        h_plot_morph(i,d) = h;
    end
end

%% 5.5. Plot wavelet coherence of activity as time series

%model = models(plot_params.idx_trial, 1);
model = models_idx(1,plot_params.idx_prob,1,1);
edges = model.edges;
vertices = model.vertices;
weight_labels = cell(N_idx_edge,1);
for i = 1 : N_idx_edge
   edge = edges(i);
   weight_labels(i) = {sprintf('%s->%s', vertices(edge.from).label, vertices(edge.to).label)};
end
N_v = length(vertices);
%weight_labels = cell(N_idx_edge,1);
ts = results(1,1).t;
N_t = length(ts);

h_plot_wcoh = zeros(N_idx_trial, N_idx_delay);
for i = 1 : N_idx_trial
    idx_i = plot_params.idx_trial(i);
    
    for d = 1 : N_idx_delay
        idx_d = plot_params.idx_delay(d);
        h = figure;
        set (h, 'Color', 'w');
        for w = 1 : N_idx_weight
            idx_w = plot_params.idx_weight(w);
            result = results_idx(i,idx_d,idx_w);
            subplot(N_idx_weight,1,w); 

            N_pair = (N_v^2-N_v)/2;
            wcoh = zeros(N_t,N_idx_edge);

            for k = 1 : N_plot_edge_idx
               idx_k = plot_edge_idx(k,:);
               from = idx_k(1);
               to = idx_k(2);
               if from > to
                  tmp = from; from = to; to = tmp;
               end
               % Map i,j to upper triangle index
               idx_e = N_units * (from-1) - (from*(from-1)/2) + to - from;
               wcoh(:,k) = squeeze(result.wcoh(:,idx_e));
            end

            hh = plot(ts, wcoh);
            set(hh,'LineWidth',plot_params.line_width);

            txt = '';
            for k = 1 : length(weight_labels)
                txt = [txt sprintf('w(%s)=%1.2f ', weight_labels{k}, weights(idx_w,k))];
            end
            if plot_params.show_axis_labels
                xlabel(txt);
            end
            if plot_params.show_titles
                if params.n_delays > 1
                    hh = suptitle(sprintf('Wavelet coherence for trial #%i (delay1=%1.2f)', idx_i, delays(idx_d)));
                else
                    hh = suptitle(sprintf('Wavelet coherence for trial #%i', idx_i));
                end
                set(hh, 'FontSize', 14);
                set(hh, 'FontWeight', 'bold');
            end
            if plot_params.show_legends && w == 1
                legend(l_names);
            end
            set(gca, 'XTick', get(gca, 'XTick'), 'XTickLabel', sprintf('%i|', get(gca, 'XTick')));
    
            resize_window( h, plot_params.ts_plot_size_multi, [100 500] );
            h_plot_wcoh(i,d) = h;
        end
    end
end

%% 5.6. Plot morphology correlations versus weight

opengl software             % Required to fix rendering bug on Windows

h_morph_corr = zeros(params.n_delays, length(axis_idx));
model = models(1,1);
%model = models_idx(idx,1,1,plot_params.idx_prob);
edges = model.edges;
vertices = model.vertices;

for d = 1 : N_idx_delay
    idx_d = plot_params.idx_delay(d);
    for i = 1 : length(axis_idx)

        idx_i = axis_idx(i);
        edge = edges(idx_i);
        x_name = sprintf('%s->%s', vertices(edge.from).label, ...
                                   vertices(edge.to).label);
        values = zeros(params.n_weights, N_plot_edge_idx + 1);
        values_std = zeros(params.n_weights, N_plot_edge_idx);
        values(:,1) = weights(:,1);
        for j = 1 : params.n_weights
            c_j = squeeze(morph_corr(:,plot_params.idx_prob,idx_d,j,:,:));
            values(j,1) = weights(j,idx_i);
            for k = 1 : N_plot_edge_idx
                idx_k = plot_edge_idx(k,:);
                %edge = edges(idx_k);
                values(j,k+1) = mean(c_j(:,idx_k(1),idx_k(2)));
                values_std(j,k) = std(c_j(:,idx_k(1),idx_k(2)));
            end
        end

        h = figure;
        set(h,'Color','w');
        x = values(:,1);
        colours = colormap('lines');
        y = values(:,2:end);
        s = values_std;
        x = repmat(x,1,size(s,2));
        %clr = colours(k,:);
        hh = plot_ci_filled(x,y,y-s,y+s,colours,0.8);
        set (hh,'LineWidth',2);
        
        if plot_params.show_legends
            legend(l_names);
        end
        if plot_params.show_titles && N_idx_delay > 1
           title(sprintf('Delay = %1.2f', delays(idx_d,idx_i))); 
        end

        if plot_params.show_axis_labels
            hh = xlabel(sprintf('Weight (%s)', x_name));
            set(hh,'FontSize',12);
            set(hh,'FontWeight','bold');
            hh = ylabel('Correlation (r) - Morphology');
            set(hh,'FontSize',12);
            set(hh,'FontWeight','bold');
        end

        h_morph_corr(d,i) = h;
    end
end

%% 5.7. Plot signal coherence versus weight

opengl software             % Required to fix rendering bug

h_wcoh_plots = zeros(params.n_delays, length(axis_idx));

for d = 1 : N_idx_delay
    idx_d = plot_params.idx_delay(d);
    for i = 1 : length(axis_idx)

        idx_i = axis_idx(i);
        edge = edges(idx_i);
        x_name = sprintf('%s->%s', vertices(edge.from).label, ...
                                                     vertices(edge.to).label);
        
        values = zeros(params.n_weights, N_plot_edge_idx + 1);
        values_std = zeros(params.n_weights, N_plot_edge_idx);
        values(:,1) = weights(:,1);
        for j = 1 : params.n_weights
            m_j = squeeze(wcoh_mean(plot_params.idx_prob,idx_d,j,:));
            s_j = squeeze(wcoh_std(plot_params.idx_prob,idx_d,j,:));
            values(j,1) = weights(j,idx_i);
            for k = 1 : N_plot_edge_idx
               idx_k = plot_edge_idx(k,:);
               from = idx_k(1);
               to = idx_k(2);
               if from > to
                  tmp = from; from = to; to = tmp;
               end
               % Map i,j to upper triangle index
               idx_e = N_units * (from-1) - (from*(from-1)/2) + to - from;
               values(j,k+1) = m_j(idx_e);
               values_std(j,k) = s_j(idx_e);
            end
        end
        
        h = figure;
        set(h,'Color','w');
        x = values(:,1);
        colours = colormap('lines');
        y = values(:,2:end);
        s = values_std;
        x = repmat(x,1,size(s,2));
        hh = plot_ci_filled(x,y,y-s,y+s,colours,0.8);
        ylim([0 1]);
        set (hh,'LineWidth',2);
        
        if plot_params.show_legends
            legend(l_names);
        end
        if plot_params.show_titles && N_idx_delay > 1
           title(sprintf('Delay = %1.2f', delays(idx_d,idx_i))); 
        end

        if plot_params.show_axis_labels
            hh = xlabel(sprintf('Weight (%s)', x_name));
            set(hh,'FontSize',12);
            set(hh,'FontWeight','bold');
            hh = ylabel('Mean wavelet coherence');
            set(hh,'FontSize',12);
            set(hh,'FontWeight','bold');
        end

        h_wcoh_plots(d,i) = h;
        
    end
    
end

%% 5.8. Plot morphology correlations versus weight & delay - surface plot

axis_idx_wt = 2;               % Indices of edge(s) whose weights to use as axis
axis_idx_d = 2;                % Indices of edge(s) whose delays to use as axis
plot_edge_idx = [1];           % Indices of edge(s) to plot 
N_plot_edge_idx = length(plot_edge_idx);
h_morph_corr_3d = zeros(N_plot_edge_idx,1);

y = delays(:,axis_idx_d);
x = weights(:,axis_idx_wt);

c_names = {};
l_names = {};

% for k = 1 : N_plot_edge_idx
%    edge = edges(plot_edge_idx(k));
%    c_names = [c_names, {sprintf('%s->%s', vertices(edge.from).label, ...
%                                              vertices(edge.to).label)}];
%    l_names = [l_names, {sprintf('%s/%s', vertices(edge.from).label, ...
%                                           vertices(edge.to).label)}];
% end

% One plot per edge
for k = 1 : N_plot_edge_idx
    idx_e = plot_edge_idx(k);
    edge = edges(idx_e);
    z = squeeze(mean(morph_corr(:,:,:,edge.from,edge.to),1));
    h = figure; 
    set(h, 'Color', 'w');
    surf(x,y,z,'FaceColor','interp','EdgeColor','none');
    view([0 90]);
   
    if plot_params.show_axis_labels
        xlabel(sprintf('Weights - %s', l_names{k}));
        ylabel(sprintf('Delays - %s (ms)', l_names{k}));
    end
    colorbar;
    if plot_params.show_titles
        hh = title('Correlation (r) - Morphology');
        set(hh,'FontSize', 14);
        set(hh,'FontWeight', 'bold');
    end
    
    h_morph_corr_3d(k) = h;
end

%% 5.9. Plot signal coherence versus weight & delay - surface plot

axis_idx_wt = 2;               % Index of edge whose weights to use as axis
axis_idx_d = 2;                % Index of edge whose delays to use as axis
plot_edge_idx = [1];           % Indices of edge(s) to plot 
N_plot_edge_idx = length(plot_edge_idx);
h_wcoh_3d = zeros(N_plot_edge_idx);

y = delays(:,axis_idx_d);
x = weights(:,axis_idx_wt);

% for k = 1 : N_plot_edge_idx
%    edge = edges(plot_edge_idx(k));
%    c_names = [c_names, {sprintf('%s->%s', vertices(edge.from).label, ...
%                                              vertices(edge.to).label)}];
%    l_names = [l_names, {sprintf('%s/%s', vertices(edge.from).label, ...
%                                           vertices(edge.to).label)}];
% end

% One plot per edge
for k = 1 : N_plot_edge_idx
    idx_e = plot_edge_idx(k);
    edge = edges(idx_e);
    from = edge.from;
    to = edge.to;
    if from > to
      tmp = from; from = to; to = tmp;
    end
    % Map i,j to upper triangle index
    idx_e = N_units * (from-1) - (from*(from-1)/2) + to - from;
   
    z = wcoh_mean(:,:,idx_e);
    h = figure; 
    set(gcf, 'Color', 'w');
    surf(x,y,z,'FaceColor','interp','EdgeColor','none');
    view([0 90]);
   
    if plot_params.show_axis_labels
        xlabel(sprintf('Weights - %s', l_names{k}));
        ylabel(sprintf('Delays - %s (ms)', l_names{k}));
    end
    colorbar;
    if plot_params.show_titles
        hh = title('Wavelet Coherence');
        set(hh,'FontSize', 14);
        set(hh,'FontWeight', 'bold');
    end
    
    h_wcoh_3d(k) = h;
end

%% 5.10. Plot graph (if applicable)

if ~isfield(model, 'graph')
   fprintf('Models do not have a graph field\n'); 
end

clear vertex_labels;
vertex_labels = cell(length(model.vertices),1);
for i = 1 : length(model.vertices)
    vertex = model.vertices(i);
    vertex_labels(i) = {vertex.label};
end

if plot_params.graph_show_arrows
    bg = biograph(model.graph,vertex_labels,'ShowArrows','on','ShowWeights','off');
else
    bg = biograph(model.graph,vertex_labels,'ShowArrows','off','ShowWeights','off');
end

dolayout(bg);
nodes = bg.Nodes;
set (nodes, 'Shape', 'circle');
scale=100;
if isfield(model, 'graph_pos')
   graph_pos = model.graph_pos;
   for i = 1 : length(nodes)
      nodes(i).Position=graph_pos(i,:)*scale; 
   end
   dolayout(bg, 'Pathsonly', true);
end

view(bg);
child_handles = allchild(0);
names = get(child_handles,'Name');
h_graph = child_handles(find(strncmp('Biograph Viewer 1', names, 17)));
set(h_graph, 'Name', sprintf('Network: Trial #%i', plot_params.idx_trial));

%% 5.11. Plot graph symmetry (if applicable)

if ~isfield(model, 'graph')
   fprintf('Models do not have a graph field\n'); 
end

cmap = jet;

clear vertex_labels;
vertex_labels = cell(length(model.vertices),1);
for i = 1 : length(model.vertices)
    vertex = model.vertices(i);
    vertex_labels(i) = {vertex.label};
end

S = compute_graph_symmetry(model.graph);

if plot_params.graph_show_arrows
    bg = biograph(S,vertex_labels,'ShowArrows','on','ShowWeights','on');
else
    bg = biograph(S,vertex_labels,'ShowArrows','off','ShowWeights','on');
end

dolayout(bg);
nodes = bg.Nodes;
set (nodes, 'Shape', 'circle');
scale=100;
if isfield(model, 'graph_pos')
   graph_pos = model.graph_pos;
   for i = 1 : length(nodes)
      nodes(i).Position=graph_pos(i,:)*scale; 
   end
   dolayout(bg, 'Pathsonly', true);
end

edges = bg.Edges;
for i = 1 : length(edges)
   edge = edges(i);
   vs = strsplit(edge.ID, ' -> ');
   v = vs{1};
   v_i=str2double(v(2:end));
   v = vs{2};
   v_j=str2double(v(2:end));
   sym = S(v_i,v_j);
   edge.LineColor=cmap(round(sym*size(cmap,1)),:);
   edge.Weight=sym;
end

view(bg);
child_handles = allchild(0);
names = get(child_handles,'Name');
h_graph_sym = child_handles(find(strncmp('Biograph Viewer 1', names, 17)));
set(h_graph_sym, 'Name', sprintf('Symmetry graph: Trial #%i', plot_params.idx_trial));


%% 6. Output

figures_dir = fullfile(params.figures_dir, params.output_file(1:end-4));
if ~exist(figures_dir, 'dir')
   mkdir(figures_dir); 
end

%% 6.1. Output results

% Input signals

for i = 1 : N_idx_trial
    idx_i = plot_params.idx_trial(i);
    % Input signal
    exportfig(h_plot_inputs(i), fullfile(figures_dir, sprintf('input_signal_t%i.png', idx_i)), ...
                  'Color','rgb');
    for d = 1 : N_idx_delay
        idx_d = plot_params.idx_delay(d);
        % Raw signal
        exportfig(h_plot_activity(i,d), fullfile(figures_dir, sprintf('activity_t%i_d%i.png', idx_i, idx_d)), ...
                  'Color','rgb','LineMode', 'scaled', 'LineWidth', 1.3);
        exportfig(h_plot_activity(i,d), fullfile(figures_dir, sprintf('activity_t%i_d%i.png', idx_i, idx_d)), ...
                  'Color','rgb');
        % Wavelet transformed activity
        exportfig(h_plot_wavelet(i,d), fullfile(figures_dir, sprintf('wavelet_activity_t%i_d%i.png', idx_i, idx_d)), ...
                  'Color','rgb');
        % Morphology
        exportfig(h_plot_morph(i,d), fullfile(figures_dir, sprintf('morphology_t%i_d%i.png', idx_i, idx_d)), ...
                  'Color','rgb');
        % Wavelet coherence
        exportfig(h_plot_wcoh(i,d), fullfile(figures_dir, sprintf('wcoh_t%i_d%i.png', idx_i, idx_d)), ...
                  'Color','rgb');
    end
end

%% 6.2. Output correlation & coherence - line plots

for d = 1 : N_idx_delay
    idx_d = plot_params.idx_delay(d);
    for i = 1 : length(axis_idx)
        idx_i = axis_idx(i);
        % Morphology correlations
        exportfig(h_morph_corr(d,i), fullfile(figures_dir, sprintf('morph_corr_d%i_ax%i', idx_d, idx_i)), ...
                  'Color','rgb');
        % Coherence
        export_fig(h_wcoh_plots(d,i), fullfile(figures_dir, sprintf('mean_wcoherence_d%i_ax%i', idx_d, idx_i)), ...
                  'Color','rgb');
    end
end

%% 6.2. Output correlation & coherence - surface plots

for k = 1 : N_plot_edge_idx
%      export_fig(fullfile(figures_dir, sprintf('morph_corr_3d_ax%i', idx_k)), h_morph_corr_3d(k));
%      export_fig(fullfile(figures_dir, sprintf('wcoh_3d_ax%i', idx_k)), h_wcoh_3d(k));
    print(h_morph_corr_3d(k), '-dpng', '-r300', fullfile(figures_dir, sprintf('morph_corr_3d_ax%i.png', idx_k)));
    print(h_wcoh_3d(k), '-dpng', '-r300', fullfile(figures_dir, sprintf('wcoh_3d_ax%i.png', idx_k)));
    
end


%% 7. Miscellaneous Plots

%% 7.1. Wavelet coherence on activity

idx_wts = [3];
idx_vert = [1 2];
N_w = length(idx_wts);

legend_text = cell(N_idx_vertex,1);
for k = 1 : N_idx_vertex
    vertex = vertices(plot_params.idx_vertex(k));
    legend_text(k) = {vertex.label};
end
inputs = model.params.input_signal;
ts = results(1,1,1).t;

h_plot_activity_wcoh = zeros(N_idx_trial,N_idx_delay);
for i = 1 : N_idx_trial
    idx_i = plot_params.idx_trial(i);
    
    for d = 1 : N_idx_delay
        idx_d = plot_params.idx_delay(d);
        h = figure;
        set (h, 'Color', 'w');
        
        % Plot activity
        for w = 1 : N_w
            idx_w = idx_wts(w);
            %result = results(idx_i,idx_d,idx_w);
            result = results(i,idx_d,idx_w);
            if N_w > 1
                subplot(N_idx_weight,1,w); 
            end
            if isempty(plot_params.line_colours)
                hh = plot(ts, result.E(:,plot_params.idx_vertex));
            else
                for a = 1 : N_units
                    hh = plot(ts, result.E(:,plot_params.idx_vertex(a)), 'Color', plot_params.line_colours(a,:));
                    hold on;
                end
                hold off;
            end
            set(gca, 'XTick', get(gca, 'XTick'), 'XTickLabel', sprintf('%i|', get(gca, 'XTick')));
    
            txt = '';
            for k = 1 : length(weight_labels)
                txt = [txt sprintf('w(%s)=%1.2f ', weight_labels{k}, weights(idx_w,k))];
            end
            if plot_params.show_axis_labels
                xlabel(txt);
            end
            ylim([0 1]);
            
             % Plot coherence
            from = idx_vert(1);
            to = idx_vert(2);
            if from > to
                tmp = from; from = to; to = tmp;
            end
            % Map i,j to upper triangle index
            idx_e = N_units * (from-1) - (from*(from-1)/2) + to - from;
            wcoh = squeeze(result.wcoh(:,idx_e));

            hold on;
            hh = plot(ts, wcoh, 'Color', 'k');
            set(hh,'LineWidth',plot_params.line_width);
            hold off;
        end
        
        % Titles, etc.
        if plot_params.show_titles
            if params.n_delays > 1
                hh = suptitle(sprintf('Simulated activity for trial #%i (delay1=%1.2f)', idx_i, delays(idx_d,1)));
            else
                hh = suptitle(sprintf('Simulated activity for trial #%i', idx_i));
            end
            set(hh, 'FontSize', 14);
            set(hh, 'FontWeight', 'bold');
        end
        if plot_params.show_legends
            legend(legend_text);
        end
        
        resize_window( h, plot_params.ts_plot_size_multi, [100 500] );
        h_plot_activity_wcoh(i,d) = h;
    end
end

%% 7.1.1. Output

exportfig(h_plot_activity_wcoh(i,d), fullfile(figures_dir, sprintf('activity_wcoh_t%i_d%i.png', idx_vert(1), idx_vert(2))), ...
                  'Color','rgb','LineMode', 'scaled', 'LineWidth', 1.3);

%% 7.2. Wavelet-transformed activity on raw activity

idx_vertex = [1];
N_v = length(idx_vertex);

legend_text = cell(N_v,1);
for k = 1 : N_v
    vertex = vertices(plot_params.idx_vertex(k));
    legend_text(k) = {vertex.label};
end
inputs = model.params.input_signal;
ts = results(1,1,1).t;

h_plot_activity_wcoh = zeros(N_idx_trial,N_idx_delay);
for i = 1 : N_idx_trial
    idx_i = plot_params.idx_trial(i);
    
    for d = 1 : N_idx_delay
        idx_d = plot_params.idx_delay(d);
        h = figure;
        set (h, 'Color', 'w');
        
        % Plot activity
        for w = 1 : length(idx_wts)
            idx_w = plot_params.idx_weight(w);
            %result = results(idx_i,idx_d,idx_w);
            result = results(i,idx_d,idx_w);
            if length(idx_wts) > 1
                subplot(N_idx_weight,1,w); 
            end
            if isempty(plot_params.line_colours)
                plot(ts, result.E(:,idx_vertex));
            else
                for a = 1 : N_v
                    plot(ts, result.E(:,idx_vertex(a)), 'Color', 'b');
                    hold on;
                end
                hold off;
            end
            hold on;
            if isempty(plot_params.line_colours)
                hh = plot(ts, result.E_wav(:,idx_vertex(a)), 'Color', 'k');
                set(hh, 'LineWidth', 2);
            else
                for a = 1 : N_v
                    hh = plot(ts, result.E_wav(:,idx_vertex(a)), 'Color', plot_params.line_colours(a,:));
                    set(hh, 'LineWidth', 2);
                    hold on;
                end
                hold off;
            end
            
            set(gca, 'XTick', get(gca, 'XTick'), 'XTickLabel', sprintf('%i|', get(gca, 'XTick')));
    
            txt = '';
            for k = 1 : length(weight_labels)
                txt = [txt sprintf('w(%s)=%1.2f ', weight_labels{k}, weights(idx_w,k))];
            end
            if plot_params.show_axis_labels
                xlabel(txt);
            end
            ylim([0 1]);
            
            
        end
        
        % Titles, etc.
        if plot_params.show_titles
            if params.n_delays > 1
                hh = suptitle(sprintf('Simulated activity for trial #%i (delay1=%1.2f)', idx_i, delays(idx_d,1)));
            else
                hh = suptitle(sprintf('Simulated activity for trial #%i', idx_i));
            end
            set(hh, 'FontSize', 14);
            set(hh, 'FontWeight', 'bold');
        end
        if plot_params.show_legends
            legend(legend_text);
        end
        
        resize_window( h, plot_params.ts_plot_size_multi, [100 500] );
        h_plot_activity_wcoh(i,d) = h;
    end
end

%% 7.2.1. Output

for i = 1 : N_v
    exportfig(h_plot_activity_wcoh(i,d), fullfile(figures_dir, sprintf('activity_wav_i%i.png', idx_vertex(i))), ...
                  'Color','rgb','LineMode', 'scaled', 'LineWidth', 1.3);
end

