function [ result ] = activity_growth_sim_thres ( initial_size, ts, activity, params, vertices )
%ACTIVITY_GROWTH_SIM Simulates activity-dependent growth of regions,
%                    defined by thresholds on activity and morphology
%   
% INPUT:
%
% initial_size - N_r x 1 vector of initial morphometry
% ts           - N_t x 1 time values
% activity     - N_t x N_r matrix of vertex-wise activity
% params:
%
%       morph_min, morph_max        - constrains morphology
%       thres_A_low, thres_A_high   - thresholds for atrophy/growth
%       tau                         - rate constant
%
% thresholds   - Optional vector of vertex-wise thresholds. Must be 
%                vertex structures with an activity member.  Otherwise, 
%                uses params.thres_A_low/high.
%
% vertices     - Optional list of vertex structs; vertices must contain an
%                activity field specifying the vertex-wise activity
%                thresholds. Otherwise, uses params.thres_A_low/high.
%

has_vertices = exist('vertices','var');
N_v = length(initial_size);

[t, morph] = ode45( @solve, ts, initial_size );

result.t = t;
result.morph = morph;

    function dydt = solve ( t, y )

        yy = zeros(N_v,1);
        for i = 1 : N_v
            A = linterp(ts, activity(:,i), t);
            if has_vertices
                vertex = vertices(i);
                thres_A_low = vertex.activity.thres_low;
                thres_A_high = vertex.activity.thres_high;
            else
                thres_A_low = params.thres_A_low;
                thres_A_high = params.thres_A_high;
            end
            
            if A < thres_A_low
                yy(i) = -(thres_A_low - A) * max(y(i) - params.morph_min, 0);
            elseif A > thres_A_high
                yy(i) = (A - thres_A_high) * max(params.morph_max - y(i), 0);
            else
                yy(i) = 0;
            end
        end
        yy = yy / params.tau;
        dydt = yy;
    end

end

