function [ result ] = activity_growth_sim_gauss ( initial_size, ts, activity, params, vertices )
%ACTIVITY_GROWTH_SIM Simulates activity-dependent growth of regions,
%                    defined by Gaussian distributions of activity and
%                    morphology
%   
% INPUT:
%
% initial_size - N_r x 1 vector of initial morphometry
% ts           - N_t x 1 time values
% activity     - N_t x N_r matrix of vertex-wise activity
% params:
%
%       fixed_distribution     - if true, used a fixed activity
%                                distribution, defined by activ_mean and
%                                activ_std; if false, uses the vertex-wise
%                                activity to determine these values
%       activ_mean, activ_std  - defines the Gaussian activity distribution
%       alpha                  - rate of growth/shrinkage
%       morph_mean, morph_std  - defines the Gaussian morphometry
%                                distribution
%

N_v = length(initial_size);

if ~params.fixed_distribution
    % Determine activity distribution from vertex-wise activity
    activ_mean = zeros(N_v,1);
    activ_std = zeros(N_v,1);
    for j = 1 : N_v
        activ_mean(j) = mean(activity(:,j));
        activ_std(j) = std(activity(:,j));
    end
else
    % Use pre-defined (fixed) activity distribution
    activ_mean = repmat(params.activ_mean,N_v,1);
    activ_std = repmat(params.activ_std,N_v,1);
end

[t, morph] = ode45( @solve, ts, initial_size );

result.t = t;
result.morph = morph;

    function dydt = solve ( t, y )

        yy = zeros(N_v,1);
        for i = 1 : N_v
            A = linterp(ts, activity(:,i), t);

            r_A     = normcdf(A, activ_mean(i), activ_std(i)) - 0.5;
            r_M     = 0.5-normcdf(y(i), params.morph_mean, params.morph_std);
            
            yy(i) = params.alpha * (r_A + params.beta * r_M);
            
        end
        dydt = yy;
    end

    function ss = sigmoid ( xx )
        ss = 1./(1 + exp(-xx));
    end

end

