function [ errors ] = get_roi_prediction_error2( Y, B, sig, type )

% Computes the prediction error (y - y_hat) from the given model (set of
% beta values B obtained from get_correlation_matrix2), for new values Y
% (N x S ROI-wise and subject-wise values as in get_correlation_matrix2). 
% 
% Y           An N x S matrix, where N is the number of subjects, and 
%             S is the number of values per subject
% B           An S x S x 2 matrix of beta values (constant + slope)
%             defining the linear regression model for ROIs i and j
% type        [Optional] One of: 
%                   'delta_y' (default): to return the y prediction
%                   errors (asymmetrical)
%                   'perp': to compute the perpendicular
%                   distances from the linear model (norm)
%
% error       An S x S n N matrix of prediction errors
%
% See also get_correlation_matrix2 get_prediction_error
%
% AUTHOR: Andrew Reid, MNI, 2011

if nargin < 3
   type = 'delta_y'; 
end

N = size(Y, 1);
S = size(Y, 2);
errors = zeros(S, S, N);

for i = 1:S
    x = [Y(:,i),ones(N,1)];
    %for j = i+1:S
    for j = 1:S
        if i ~= j
            if sig(i,j)
                y = Y(:,j);
                b = squeeze(B(i,j,:));
                error=get_prediction_error(x,y,b);
                errors(i,j,:) = error;
            end
        end
    end
end

if strcmp(type, 'delta_y')
    return;
end

% Compute perpendicular distance
for i = 1:S
    for j = i+1:S
        if sig(i,j)
            y1 = errors(i,j,:);
            y2 = errors(j,i,:);
            delta=y2./y1;
            thetas = atan(delta);
            deltas = sin(thetas).*y1;
            errors(i,j,:)=abs(deltas);
            errors(j,i,:)=abs(deltas);
        end
    end
end
