function [ C, P, E ] = get_partial_correlation_matrix( Y, threshold, do_fdr )
% GET_CORRELATION_MATRIX() Computes a correlation matrix C from the N x S
% input matrix Y, and threshold using p and (optionally) FDR q values. Also
% optionally returns the effect sizes (slopes)
%
% PARAMETERS:
%   Y           An N x S matrix, where N is the number of values per
%               subject, and S is the number of subjects
%   threhsold   [optional] Threshold for p and q values (default = 0.05)
%   do_fdr      [optional] If true, apply an FDR threshold (default =
%               false)
%
% RETURNS:
%   C           An N x N matrix of correlation coefficients, thresholded by
%               p- and optionally q-values
%   P           The p-values
%
%   E           Matrix of effect sizes (slopes)
%
% AUTHOR: Andrew Reid, MNI, 2011

if nargin < 2 || isempty(threshold)
   threshold = 0.05; 
end

if nargin < 3 || isempty(do_fdr)
   do_fdr = false; 
end

N = size(Y,1);
S = size(Y,2);

% Get coefficients
% [C, P] = corrcoef(Y);
%C = gretna_parcorr(Y);
[C, P] = partialcorr(Y);

E = zeros(S,S);

for i = 1:S
    x = Y(:,i);
    for j = i+1:S
        y = [Y(:,j),ones(N,1)];
        b=regress(x,y);
        E(i,j)=b(1);
        E(j,i)=b(1);
    end
end

% Threshold with P-values
C = C.*double(P < threshold);
E = E.*double(P < threshold);

% FDR threshold, if necessary
if do_fdr
   C = C.*double(fdr_bky(P,threshold)); 
   E = E.*double(fdr_bky(P,threshold)); 
end

end

