function [p R] = get_partial_correlation(A)
% *************************************************************************
%   fmri_ParCorr: Partial Correlation Matrix
%   [R] = gretna_parcorr(A) 
%   Input:
%       A   - each row is an observation and each column is  a variable. 
%              Namely, A is a matrix of column vectors (i.e. A = [A1 A2 ... An]
%              where An is a column vector).
%   Output:
%       R   - partial correlation matrix
%       P   - corresponding p-values
%
%   He Yong, BIC,MNI,McGill
%   2007/02/01
%   Andrew Reid, MNI (Added p-value computation from corrcoef)
%   2011
% *************************************************************************


if nargin >= 2  
    error('error input parameters.');end

B = inv(cov(A));

R = zeros(size(B));
for i = 1:size(B,1)
  for j = 1:size(B,2)
    R(i,j) = - B(i,j) / sqrt(B(i,i) * B(j,j));
  end
end

% Significance

[n,m] = size(B);

% Compute p-value if requested.
if nargout>=2
   % Operate on half of symmetric matrix.
   lowerhalf = (tril(ones(m),-1)>0);
   rv = R(lowerhalf);
   if length(n)>1
      nv = n(lowerhalf);
   else
      nv = n;
   end

   % Tstat = +/-Inf and p = 0 if abs(r) == 1, NaN if r == NaN.
   Tstat = rv .* sqrt((nv-2) ./ (1 - rv.^2));
   p = zeros(m,class(B));
   p(lowerhalf) = 2*tpvalue(-abs(Tstat),nv-2);
   p = p + p' + diag(diag(R)); % Preserve NaNs on diag.

end

% ------------------------------------------------
function p = tpvalue(x,v)
%TPVALUE Compute p-value for t statistic.

normcutoff = 1e7;
if length(x)~=1 && length(v)==1
   v = repmat(v,size(x));
end

% Initialize P.
p = NaN(size(x));
nans = (isnan(x) | ~(0<v)); % v == NaN ==> (0<v) == false

% First compute F(-|x|).
%
% Cauchy distribution.  See Devroye pages 29 and 450.
cauchy = (v == 1);
p(cauchy) = .5 + atan(x(cauchy))/pi;

% Normal Approximation.
normal = (v > normcutoff);
p(normal) = 0.5 * erfc(-x(normal) ./ sqrt(2));

% See Abramowitz and Stegun, formulas 26.5.27 and 26.7.1.
gen = ~(cauchy | normal | nans);
p(gen) = betainc(v(gen) ./ (v(gen) + x(gen).^2), v(gen)/2, 0.5)/2;

% Adjust for x>0.  Right now p<0.5, so this is numerically safe.
reflect = gen & (x > 0);
p(reflect) = 1 - p(reflect);

% Make the result exact for the median.
p(x == 0 & ~nans) = 0.5;
