function [ C, P ] = get_correlation_matrix_subj_resids( Y_pred_err, X, threshold, do_fdr )
% get_correlation_matrix_subj_resids() Computes a correlation matrix C for
% each prediction error value in the S x S x N matrix Y_pred_err, with a
% set of output values X
%
% PARAMETERS:
%   Y_pred_err  An S x S x N matrix, where N is the number of subjects, and 
%               S is the number of values per subject
%   X           Output values, of size S x M
%   threshold   [optional] Threshold for p and q values (default = 0.05)
%   do_fdr      [optional] If true, apply an FDR threshold (default =
%               false)
%
% RETURNS:
%   C           An S x S x M matrix of correlation coefficients, thresholded by
%               p- and optionally q-values
%
%   P           The corresponding p-values
%
% AUTHOR: Andrew Reid, MNI, 2011

if nargin < 2 || isempty(threshold)
   threshold = 0.05; 
end

if nargin < 3 || isempty(do_fdr)
   do_fdr = false; 
end

M = size(X,2);
S = size(Y_pred_err,1);

% Get coefficients
C = zeros(S,S,M);
P = zeros(S,S,M);

if M > 1
    for m = 1:M
        for i = 1:S
            for j = i+1:S
               [C(i,j,m),P(i,j,m)]=corrcoef(Y_pred_err(i,j,:),X(:,m));
               C(j,i,m)=C(i,j,m);
               P(j,i,m)=P(i,j,m);
            end
        end
    end

    for m = 1:M
        % Threshold with P-values
        C(:,:,m) = C(:,:,m).*double(P < threshold);

        % FDR threshold, if necessary
        if do_fdr
           C(:,:,m) = C(:,:,m).*double(fdr_bky(P,threshold)); 
        end
    end

else
    
    for i = 1:S
        for j = i+1:S
           T=squeeze(Y_pred_err(i,j,:));
           [c,p]=corrcoef(T,X);
           C(i,j)=c(1,2);
           P(i,j)=p(1,2);
           C(j,i)=c(1,2);
           P(j,i)=p(1,2);
        end
    end
    
    C(isnan(C)) = 0;
    P(isnan(P)) = 1;
    
    % Threshold with P-values
    C = C.*double(P < threshold);

    % FDR threshold, if necessary
    if do_fdr
       C = C.*double(fdr_bky(P,threshold)); 
    end
    
end

end

