function [ C, P, B, R ] = get_correlation_matrix2( Y, threshold, do_fdr, rem_outliers, alpha )
% GET_CORRELATION_MATRIX() Computes a correlation matrix C from the N x S
% input matrix Y, and threshold using p and (optionally) FDR q values. Also
% optionally returns the effect sizes (betas)
%
% PARAMETERS:
%   Y           An N x S matrix, where N is the number of subjects, and 
%               S is the number of values per subject
%   threshold   [optional] Threshold for p and q values (default = 0.05)
%   do_fdr      [optional] If true, apply an FDR threshold (default =
%               false)
%
%   remove_outliers     [optional] If true, outliers are removed which are
%                       beyond the specified alpha, prior to computing each
%                       correlation
%
%   alpha               [optional] Alpha value for outlier removal. Default
%                       is 0.05
%
% RETURNS:
%   C           An S x S matrix of correlation coefficients, thresholded by
%               p- and optionally q-values
%
%   P           The corresponding p-values
%
%   B           S x S x 2 matrix of corresponding effect sizes (betas)
%
%   R           S x S x N matrix of residuals for each pairwise regression
%
% AUTHOR: Andrew Reid, MNI, 2011

if nargin < 2 || isempty(threshold)
   threshold = 0.05; 
end

if nargin < 3 || isempty(do_fdr)
   do_fdr = false; 
end

if nargout < 3
   do_regress = false; 
else
   do_regress = true;
end

if ~exist('rem_outliers','var')
   rem_outliers = false; 
end

if ~exist('alpha','var')
   alpha = 0.05; 
end

N = size(Y,1);
S = size(Y,2);

% Get coefficients
if ~rem_outliers
    [C, P] = corrcoef(Y);
else
    Y = remove_outliers(Y,alpha,true);
    %Y = deleteoutliers(Y,alpha,true);
    [C, P] = corrcoef(Y, 'rows', 'pairwise');
end

B = zeros(S,S,2);
R = zeros(S,S,N);

if do_regress
    for i = 1:S
        x = [ones(N,1),Y(:,i)];
        for j = 1:S
            if i ~= j
            y = Y(:,j);
            [b,~,resids]=regress(y,x);
            B(i,j,:)=b;
            R(i,j,:)=resids;
            end
        end
    end
    B(:,:,1) = B(:,:,1).*double(P < threshold);
    B(:,:,2) = B(:,:,2).*double(P < threshold);
    if do_fdr
        B(:,:,1) = B(:,:,1).*double(fdr_bky(P,threshold));
        B(:,:,2) = B(:,:,2).*double(fdr_bky(P,threshold));
    end
else
    B=[];
    R=[];
end

% Threshold with P-values
C = C.*double(P < threshold);

% FDR threshold, if necessary
if do_fdr
   C = C.*double(fdr_bky(P,threshold)); 
end

end

