function [ C, P ] = get_correlation_matrix( Y, threshold, do_fdr, hasnans )
% GET_CORRELATION_MATRIX() Computes a correlation matrix C from the N x S
% input matrix Y, and threshold using p and (optionally) FDR q values
%
% PARAMETERS:
%   Y           An S x N matrix, where N is the number of values per
%               subject, and S is the number of subjects
%   threhsold   [optional] Threshold for p and q values (default = 0.05)
%   do_fdr      [optional] If true, apply an FDR threshold (default =
%               false)
%   has_nans    Evaluates corrcoef 'pairwise'
%
% RETURNS:
%   C           An N x N matrix of correlation coefficients, thresholded by
%               p- and optionally q-values
%   P           The p-values
%
% AUTHOR: Andrew Reid, MNI, 2011

if ~exist('hasnans','var')
   hasnans = false; 
end

if nargin < 2 || isempty(threshold)
   threshold = 0.05; 
end

if nargin < 3 || isempty(do_fdr)
   do_fdr = false; 
end

N = size(Y,1);
S = size(Y,2);

% Get coefficients
if hasnans
    [C, P] = corrcoef(Y, 'rows', 'pairwise');
else
    [C, P] = corrcoef(Y);
end

% Threshold with P-values
C = C.*double(P < threshold);

% FDR threshold, if necessary
if do_fdr
   C = C.*double(fdr_bky(P,threshold)); 
end

end

