function [ C, P, B, R ] = get_2var_correlation_matrix( Y1, Y2, threshold, do_fdr, rem_outliers, alpha, fdr_method )
% GET_2VAR_CORRELATION_MATRIX() Computes a correlation matrix C from two N x S
% input matrices Y1 and Y2, correlating each column of Y1 with each column of Y2. 
% Thresholds using p and (optionally) FDR q values. Also
% optionally returns the effect sizes (betas)
%
% PARAMETERS:
%   Y1,Y2       N x S matrices, where N is the number of subjects, and 
%               S is the number of values per subject
%   threshold   [optional] Threshold for p and q values (default = 0.05)
%   do_fdr      [optional] If true, apply an FDR threshold (default =
%               false)
%
%   remove_outliers     [optional] If true, outliers are removed which are
%                       beyond the specified alpha, prior to computing each
%                       correlation
%
%   alpha               [optional] Alpha value for outlier removal. Default
%                       is 0.05
%
%   fdr_method          Method used for FDR; see niak_fdr. Default is LSL
%
% RETURNS:
%   C           An S x S matrix of correlation coefficients, thresholded by
%               p- and optionally q-values
%
%   P           The corresponding p-values
%
%   B           S x S x 2 matrix of corresponding effect sizes (betas)
%
%   R           S x S x N matrix of residuals for each pairwise regression
%
% AUTHOR: Andrew Reid, MNI, 2011

if nargin < 3 || isempty(threshold)
   threshold = 0.05; 
end

if nargin < 4 || isempty(do_fdr)
   do_fdr = false; 
end

if nargin < 7
   fdr_method = 'LSL'; 
end

if nargout < 4
   do_regress = false; 
else
   do_regress = true;
end

if ~exist('rem_outliers','var')
   rem_outliers = false; 
end

if ~exist('alpha','var')
   alpha = 0.05; 
end

N = size(Y1,1);
S = size(Y1,2);

% Get coefficients
if rem_outliers
    Y1 = remove_outliers(Y1,alpha,true);
    Y2 = remove_outliers(Y2,alpha,true);
end

C = zeros(S);
P = ones(S);
for i = 1 : S
    for j = 1 : S
        if ~rem_outliers
            [Cij,Pij] = corrcoef(Y1(:,i),Y2(:,j));
        else
            [Cij,Pij] = corrcoef(Y1(:,i),Y2(:,j),'rows','pairwise');
        end
        C(i,j) = Cij(1,2);
        P(i,j) = Pij(1,2);
    end
end

if do_fdr
   [~,sig] = niak_fdr(P,fdr_method);
   sig = sig & P < 0.05;
   C = C.*double(sig);
end

B = zeros(S,S,2);
R = zeros(S,S,N);

if do_regress
    for i = 1:S
        x = [ones(N,1),Y1(:,i)];
        for j = 1:S
            if i ~= j
            y = Y2(:,j);
            [b,~,resids]=regress(y,x);
            B(i,j,:)=b;
            R(i,j,:)=resids;
            end
        end
    end
    B(:,:,1) = B(:,:,1).*double(P < threshold);
    B(:,:,2) = B(:,:,2).*double(P < threshold);
    if do_fdr
        B(:,:,1) = B(:,:,1).*double(fdr_bky(P,threshold));
        B(:,:,2) = B(:,:,2).*double(fdr_bky(P,threshold));
    end
else
    B=[];
    R=[];
end

% Threshold with P-values
C = C.*double(P < threshold);

% FDR threshold, if necessary
if do_fdr
   C = C.*double(fdr_bky(P,threshold)); 
end

end

