function [ results ] = evaluate_watts_strogatz_model( N, K, M, B, method, p_exp )
%EVALUATE_COST_MODEL Computes weighted efficiency values for a network of 
%                    size N using the Watt-Strogatz samll-world model. 
%
% PARAMETERS:
%
%   N          Size of the network (vertex count)
%   K          Number of edges per vertex
%   M          Total steps (including zero and one)
%   B          Number of samples to use
%   METHOD     [optional] Efficiency method to use; one of 'binary'
%              (GRETNA) and 'weighted' (default)
%   P_EXP      Exponent for P; change this to facilitate a better
%              distribution for semi-log plots, etc. Default = 1.
%

if nargin < 5
   method = 'weighted'; 
end

if nargin < 5
   p_exp = 1; 
end

regular = get_regular_graph(N, K);

results.p = zeros(M,1);
results.e_global = zeros(B,M,1);
results.e_local = zeros(B,M,1);

for b = 1 : B

    if strcmp(method,'weighted')
        G = rand(N).*regular;       %random weights
    else
        G=regular;
    end
    i=1;

    for p = 0 : 1 / (M - 1) : 1
        
        p2 = p ^ p_exp;  % facilitate a log plot
        
        % Rewire if p_test < p
        for v = 1 : N
           edges = find(regular(v,:) == 1);
           no_edges = find(regular(v,:) == 0);
           Ne = length(edges);
           Nn = length(no_edges);
           
           for j = 1 : Ne
              
               % If random number is smaller than current p, rewire
               if rand < p2
                   value = G(v,edges(j));
                   G(v,edges(j)) = 0;
                   index = ceil(rand * (Nn - 1)) + 1;
                   while no_edges(index) == v
                       index = ceil(rand * (Nn - 1)) + 1;
                   end
                   G(v,no_edges(index)) = value;
               end
               
           end
            
        end

        if b == 1
            results.p(i) = p2;
        end
        
        if strcmp(method,'weighted')
            eff = compute_weighted_efficiency( G, 0, 10 );
            results.e_global(b,i) = eff.e_global;
            results.e_local(b,i) = eff.e_local;
        else
            eff=gretna_efficiency(G);
            results.e_global(b,i) = eff.gE;
            results.e_local(b,i) = eff.locE;
        end

        i = i + 1;

    end

end

results.mean_e_global = mean(results.e_global,1)';
results.mean_e_local = mean(results.e_local,1)';

end

