function [ W ] = create_weighted_graph( R, beta, type, hard_threshold )
% CREATE_WEIGHTED_GRAPH (R, BETA, TYPE) constructs a weighted graph from 
% the input N x N matrix R, using a "soft-thresholding" approached, 
% detailed in Schwartz & McConigle (Neuroimage, 55 (2011) 1132-1146).
%
% PARAMETERS:
%   R - An N x N matrix of values (e.g., correlation coefficients). R must
%       be defined in the range [-1,1].
%   BETA - An exponent to apply to the thresholding, where BETA=1 is
%          linear, and higher values of BETA increase the eccentricity of
%          the mapping from R -> W
%   TYPE [optional] - Defines the type of function to apply to R; one of:
%           'positive' - Squashes negative values, amplifies positive
%                        (default behaviour)
%           'negative' - Squashes positive values, amplifies positive
%           'zero' - Squashes small magnitudes, amplifies larger,
%                    regardless of sign
%   HARD_THRESHOLD [optional] - Defines a hard threshold on the graph,
%                               which is a minimum for type 'positive', 
%                               maximum for type 'negative' and absolute 
%                               minimum for type 'zero'
% 
% RETURNS:
%   W - an N x N adjacency matrix of weights representing a directed graph,
%       with the range [0,1] if TYPE = 'positive', [-1,0] if TYPE =
%       'negative', or [-1,1] if TYPE = 'zero'.
%
% AUTHOR: Andrew Reid, MNI, 2011
%

if nargin < 3 || isempty(type)
   type = 'positive'; 
end

if strcmp(type,'positive')
   W = ((R + 1)./2).^beta;
   if exist('hard_threshold')
      W = W.*double(W > hard_threshold); 
   end
   return;
end

if strcmp(type,'negative')
   W = -((((-R) + 1)./2).^beta);
   if exist('hard_threshold')
      W = W.*double(W < hard_threshold); 
   end
   return;
end

if strcmp(type,'zero')
   W = R.^beta;
   if exist('hard_threshold')
      W = W.*double(abs(W) > hard_threshold); 
   end
   return;
end

error('CreateWeightedGraph:BadType', 'Bad type defined; see the help printout for valid values.');

end

