function [ results ] = compute_weighted_efficiency( W, norm_min, norm_max )
% COMPUTE_WEIGHTED_EFFICIENCY (W) computes global and local efficiency
% measures for W, an N x N adjacency matrix representing a weighted graph. 
% See Latora & Machiori (Eur Phys J, 2003) for a detailed general 
% description of this computation.
%
% Results are normalized to the ideal (maximum) efficiencies.
%
% PARAMETERS:
%   W - an N x N adjacency matrix representing a weighted graph. W must 
%       contain only positive or zero elements.
%
%   norm_min, norm_max - Min and max defining a normalization range for the
%                        weights. The same range should be used if weights
%                        are to be compared, and should be large enough to
%                        encompass all weights. The result will be relative
%                        to this range.
%
% RETURNS:
%   results.e_global - The global efficiency of W
%   results.e_global_regional - The global efficiency at each node
%   results.e_local - The average local efficiency of W
%   results.e_local_regional - The local efficiency at each node
%
% AUTHOR: Andrew Reid, MNI, 2011
%         Adapted from Yong He, MNI, 2007

if nargin < 2
   norm_min = 0;
   norm_max = 1;
end

% Since correlation coefficients are inverse distance measures,
% normalize and invert to compute weighed efficiency
W = normalize_data2(W, 0, 1, norm_min, norm_max);
W = 1 - W;

W(W == 1) = Inf;    % Non-existence paths have distance of Inf

% Compute all shortest paths i,j using Floyd-Warshall algorithm
N = length(W);
W_sparse=sparse(W);
D = floyd_warshall_all_sp( W_sparse );

% Set diagonals (self-self distances) to Infinity
%D = D + diag(diag(1./zeros(N,N)));

% Invert distances
%D = 1./D;
D = normalize_data2(D, 0, 1, norm_min, norm_max);
D = 1 - D;
D = D - eye(N);

D(isinf(D)) = 0;    % Necessary for disconnected graphs

% Compute global efficiency
results.e_global = (sum(D(:)))/(N*(N-1));

% Normalize to the ideal global efficiency
% W_ideal = zeros(N);
% e_ideal = (sum(W_inv(:)))/(N*(N-1));
% results.e_global = results.e_global/e_ideal;
results.e_global_regional = sum(D,2)/(N-1);
results.e_local_regional = zeros(N,1);

% Compute average local efficiency
for i = 1:N
    NV =  find(W(i,:)~=Inf);
    if isempty(NV) || length(NV)==1
        results.e_local_regional(i) = 0;
    else
        N2 = length(NV);
        B = W(NV,NV);
        [D1] = floyd_warshall_all_sp(sparse(B));
        %D1 = D1 + diag(diag(1./zeros(size(D1))));
        %D1 = 1./D1;
        D1 = normalize_data2(D1, 0, 1, norm_min, norm_max);
        D1 = 1 - D1;
        D1 = D1 - eye(N2);
        
        D1(isinf(D1)) = 0;     % Necessary for disconnected graphs
         
        results.e_local_regional(i)  = sum(D1(:))/(N2*(N2-1)) ;
%       e_ideal = sum(B(:))/(N2*(N2-1));
        results.e_local_regional(i) = results.e_local_regional(i);
    end
end

results.e_local = sum(results.e_local_regional(:)) / N;

end

