function [ results ] = bootstrap_graphs_pcorr( Y, iter, sample_size, do_fdr )
% BOOTSTRAP_GRAPHS_PCORR(Y, ITER, SAMPLE_SIZE) takes an N x S matrix Y 
% of observed values and constructs ITER network graphs using partial 
% correlations, where :
%       N = the number of values per subject
%       S = the number of subjects
%       each iteration operates on a random sampling (with replacement) of 
%           Y of size SAMPLE_SIZE.
%
% PARAMETERS:
%       Y - The observed values, N x S matrix
%       ITER - The desired number of bootstrap samples
%       SAMPLE_SIZE [optional] - The size of each bootstrap sample (default
%       is S)
%
% RETURNS:
%       results.r_boot - An N x N x SAMPLE_SIZE matrix containing the 
%                        partial correlation coefficients for each 
%                        bootstrap sample
%       results.p_boot - An N x N x SAMPLE_SIZE matrix containing the 
%                        corresponding P values
%       results.fdr_boot - An N x N x SAMPLE_SIZE logical matrix specifying
%                          which P-values survive FDR correction at q=0.05
%
% AUTHOR: Andrew Reid, MNI, 2011
%

N = size(Y,1);
S = size(Y,2);

if nargin < 3 || isempty(sample_size)
   sample_size = S; 
% elseif sample_size > S
%    sample_size = S;
end

if nargin < 4
   do_fdr = false; 
end

indices = 1:S;

% Get random sampling indices
S_samp = zeros(iter,sample_size) ;
for i=1:iter;
    S_samp(i,:) = randsample(indices,sample_size,'true') ;
end
S_samp = S_samp' ;

% Initiate results
Y_boot = zeros(sample_size,N,iter) ;
results.r_boot = zeros(N,N,iter) ;
results.p_boot = zeros(N,N,iter) ;
results.e_boot = zeros(N,N,iter) ;
results.fdr_boot = zeros(N,N,iter) ;

% Create sampling matrices
for k=1:iter
    for i=1:sample_size
        Y_boot(i,:,k) = Y(:,S_samp(i,k))' ;
    end
end

% Create bootstrap samples and compute corresponding correlation 
% coefficients
for i=1:iter
    Y_sample = Y_boot(:,:,i) ;
%     [r_temp,p_temp] = corrcoef(Y_sample) ;
    [r_temp, p_temp] = partialcorr(Y_sample, 'rows', 'complete');
    % NaN's should be zeros... 
    r_temp = r_temp - eye(N,N) ;
    r_temp(isnan(r_temp)) = 0;
    p_temp(isnan(p_temp)) = 1;
    
    results.p_boot(:,:,i) = p_temp ;
    
    fdr = fdr_bky(p_temp,0.05);
    if do_fdr
        r_temp=r_temp.*double(fdr);
%         e_temp=e_temp.*double(fdr);
    end
    results.fdr_boot(:,:,i) = fdr;
    results.r_boot(:,:,i) = r_temp ;
%     results.e_boot(:,:,i) = e_temp ;
end


end

