function [ results, S_samp ] = bootstrap_graphs( Y, iter, sample_size, do_fdr, fdr_thres, type, verbose, sample )
% BOOTSTRAP_GRAPHS(Y, ITER, SAMPLE_SIZE) takes an N x S matrix Y 
% of observed values and constructs ITER network graphs, where :
%       N = the number of values per subject
%       S = the number of subjects
%       each iteration operates on a random sampling (with replacement) of 
%           Y of size SAMPLE_SIZE.
%
% PARAMETERS:
%       Y - The observed values, N x S matrix
%       ITER - The desired number of bootstrap samples
%       SAMPLE_SIZE [optional] - The size of each bootstrap sample (default
%       is S)
%       TYPE - One of 'pearson' (default) and 'partial'
%
% RETURNS:
%       results.r_boot - An N x N x ITER matrix containing the 
%                        Pearson correlation coefficients for each 
%                        bootstrap sample
%       results.p_boot - An N x N x ITER matrix containing the 
%                        corresponding P values
%       results.fdr_boot - An N x N x ITER logical matrix specifying
%                          which P-values survive FDR correction at q=0.05
%       results.fdr_crit - A ITER x 1 matrix containing
%                          the critical p value for each element
%
%       S_samp          - The random sample for the bootstraps
%
% AUTHOR: Andrew Reid, MNI, 2011
%

N = size(Y,1);
S = size(Y,2);

if nargin < 3 || isempty(sample_size)
   sample_size = S; 
end

if nargin < 4
   do_fdr = false; 
end

if nargin < 5
   fdr_thres = 0.05; 
end

if ~exist('verbose','var')
   verbose = false; 
end

indices = 1:S;

% Get random sampling indices
S_samp = zeros(iter,sample_size) ;

% If sample is predefined
if exist('sample','var')
    S_samp = sample;
else
    for i=1:iter;
        try
            S_samp(i,:) = randsample(indices,sample_size,'true') ;
        catch
           wtf=1; 
        end
    end
    S_samp = S_samp' ;
end

% Initiate results
Y_boot = zeros(sample_size,N,iter) ;
results.r_boot = zeros(N,N,iter) ;
results.p_boot = zeros(N,N,iter) ;
%results.e_boot = zeros(N,N,iter) ;
results.fdr_boot = zeros(N,N,iter) ;

% Create sampling matrices
for k=1:iter
    for i=1:sample_size
        Y_boot(i,:,k) = Y(:,S_samp(i,k))' ;
    end
end

% Create bootstrap samples and compute corresponding correlation 
% coefficients
for i=1:iter
    Y_sample = Y_boot(:,:,i) ;
    if exist('type','var') && strcmp(type,'partial')
        [r_temp, p_temp] = partialcorr(Y_sample, 'rows', 'complete');
    else
        [r_temp, p_temp] = get_correlation_matrix2(Y_sample,fdr_thres,do_fdr);
    end
    r_temp = r_temp - eye(N,N) ;
    results.r_boot(:,:,i) = r_temp ;
    results.p_boot(:,:,i) = p_temp ;
    [results.fdr_boot(:,:,i),results.fdr_crit(i)] = fdr_bky(p_temp,fdr_thres);
    if verbose
       fprintf('%s %i\n', 'Completed bootstrap ', i);
    end
end


end

