function [ results, residuals ] = analyze_subject_edge_residuals_glmnet_permtest(y, x, ref_filter, params, y_filter)

% Obtains residuals (prediction errors y_hat - y), and
% uses these as factors in a glmnet design to obtain the strongest
% predictors of some score x. Then runs a permutation test
%
% PARAMETERS:
% 
% y             The y scores (e.g., cortical thickness or residuals after 
%               removing covariates); an N x S matrix, 
%               where S is the number of subjects, and N is the number of 
%               values per subject
% x             The x scores to predict with the residuals defined by y and
%               model; a vector of size S
% ref_filter    A vector of size S defining the subjects from which to
%               obtain a linear model; residuals will be computed relative
%               to this model. E.g., you may want to use a normal control
%               groups as this reference model.
% type          The type of residual to compute; see
%               @get_roi_prediction_error; default is 'perp'
% params        Optional parameters as a struct, with fields:
%
%       p_threshold         p- and q-value threshold for the reference model; 
%                           default is 0.05    
%       do_fdr              Whether to correct for multiple comparisons
%                           using FDR; default is true
%       abs_resid           Whether to use the absolute value of the
%                           residuals; default is true
%       alphas              Alpha values to use for the glmnet prediction;
%                           determines the beta penalty method, on a
%                           continuum from ridge regression (alpha=0) to
%                           lasso (alpha=1); default is 0:0.25:1. See 
%                           glmnet for details
%       niters              Number of iterations for permutation test.
%
%       [glmnetSet parameters]
%
% y_filter      Filter for the y data set; default is no filter
%
% RETURNS:
%
% results       A struct containing the output of the analysis...
%
% AUTHOR: Andrew Reid, MNI, 2011

% Obtain model

if nargin < 4
   params.p_threshold = 0.05;
   params.do_fdr = true;
   params.abs_resid = true;
   params.alphas=0:0.25:1;
   params.nfolds=50;
   params.foldid=[];
   params.verbose=true;
   params.resid_type = 'perp'; 
end

N=size(y,1);
S=size(y,2);

% Get y values for reference model
y_model=y(:,ref_filter)';

% Compute intercepts + betas
[ ~, ~, betas, ~ ] = get_correlation_matrix2( y_model, params.p_threshold, params.do_fdr );

% Compute prediction error (y - y_hat)
if exist('y_filter','var')
   y = y(:,y_filter);
   S=size(y,2);
end
y_pred_err=get_roi_prediction_error(y',betas, params.resid_type);

% Take absolute value if desired
if params.abs_resid, y_pred_err = abs(y_pred_err); end

% Compute glmnet cross-validation results

% Use only top diagonal to predict
N = size(y_pred_err,1);
y_top=get_top(y_pred_err);
%results=glmnet_crossval(y_top', x, params);
%results = cell(numel(params.alphas),1);
for i = 1:numel(params.alphas)
    
    alpha = params.alphas(i);
    params.alpha = alpha;
    if params.verbose
       figure; 
    end
    result = glmnet(y_top',x,'gaussian', params);

    params.lambda=result.lambda;
    
    % run permutation tests here
    perm_betas=zeros(size(result.beta));
    for j = 1:params.niters
        S_y = size(y_top);
        N_y = prod(S_y);
        y_rnd=reshape(y_top,N_y,1);
        y_rnd=randsample(y_rnd,N_y);
        y_rnd=reshape(y_rnd,S_y);
        this_result=glmnet(y_rnd',x,'gaussian', params);
        perm_betas=perm_betas+this_result.beta;
        fprintf('%s','.');
    end
    
    % compare sample betas with random betas
    p_means=mean(perm_betas);
    p_std=std(perm_betas);
    for j = 1 : size(perm_betas,2)
        % Make two-tailed
        tmp=result.beta(:,j);
        p = normcdf(tmp,p_means(j),p_std(j));
        a = p;
        a(a>0.5)=0.5;
        a = 2*a;
        p = a - p;
        p = p * 2;
        permtest_p(:,j)=p;
        
    end
    
    w.glmnet_object = result;
    w.beta=put_bottom_back_on(result.beta,N);
    w.permtest_p = put_bottom_back_on(permtest_p,N);
    results(i)=w;
    
end

residuals=y_top;
%clear('betas');

% 
%  for i = 1 : length(results)
%     %betas(i,:,:)=results(i).glmnet_object.beta;
% %     beta=results(i).glmnet_object.beta;
% %     wtf=struct();
% %     wtf.beta=put_bottom_back_on(beta,N);
% % %     beta_stats.mean=put_bottom_back_on(beta_stats.mean,N);
% % %     beta_stats.std=put_bottom_back_on(beta_stats.std,N);
% %     betas(i).beta=beta;
% %     f(i)=wtf;
%  end

%betas=f;


end

function [ top ] = get_top (matrix)

    N = size (matrix,1);
    S = size (matrix,3);
    M = (N * (N - 1)) / 2;
    top = zeros(M,S);

    k = 1;
    for i = 2:N
        for j=i+1:N
            top(k,:) = matrix(i,j,:);
            k = k + 1;
        end
    end

end

function [ whole ] = put_bottom_back_on (top, N)

    S = size (top,2);
    whole = zeros(N,N,S);

    k = 1;
    for i = 1:N
        for j=i+1:N
            whole(i,j,:) = top(k,:);
            whole(j,i,:) = top(k,:);
            k = k + 1;
        end
    end

    a=0;
    
end
