function [ results ] = analyze_corr_net_binary ( X, groups, group_list, params, boots );

% Analyzes regional thickness values by generating bootstrap samples (with
% replacement) and computing graph efficiency measures on each sample. This
% method generates binary graphs using a range of threshold values,
% specified by the params argument. Only positive values are used for graph
% construction.
%
% Thresholds can be 'sparsity' or 'pearson' coefficients.
%
% PARAMETERS:
%   
%  X                        ROI-wise thickness values (or residuals); R x N
%                           where N = no. subjects, R = no. ROIs
%  groups                   N x 1 list of group assignments, as cell
%                           strings
%  group_list               G x 1 list of groups to analyze
%  params                   Parameters defining the analysis; see
%                           analyze_corr_net_binary_params
%
% RETURNS:
%   A result structure containing all bootstrap results, as well as mean,
%   standard deviation, and integrals calculated across a threshold range
%   (this is the entire range unless specified by the params argument).
%
% Andrew Reid
% Montreal Neurological Institute
% 2011

thr_min = 0.05;
thr_max = 0.4;
thr_step = 0.025;
boot_n = 20;
boot_m = 100;
int_start = 0.1;
int_end = 0.3;
integral_type='trap';
remove_neg=true;
show_plots=true;
keep_boots=false;
threshold_by='sparsity';
fdr_thres=0.05;
fdr_type='normal';

if nargin > 3
    thr_min = params.thr_min;
    thr_max = params.thr_max;
    thr_step = params.thr_step;
    boot_n = params.boot_n;
    boot_m = params.boot_m;
    int_start = params.int_start;
    int_end = params.int_end;
    integral_type=params.integral_type;
    remove_neg=params.remove_neg;
    show_plots=params.show_plots;
    keep_boots=params.keep_boots;
    threshold_by=params.threshold_by;
    fdr_thres=params.fdr_thres;
    fdr_type=params.fdr_type;
end

int_s = -1;
int_e = -1;
s = thr_min;
j = 0;

while s <= thr_max
    j = j + 1;
    if s >= int_start && int_s < 0
       int_s = j; 
    end
    if s >= int_end && int_e < 0
       int_e = j - 1; 
    end
    s = s + thr_step;
end

thr_count = j;

% Bootstrap with residuals:
% 1. Define a sparsity range + interval
results.X=X;
results.params=params;

elapsed = 0;

G = length(group_list);
for i = 1 : G
    % Get values for this group
    group = cell2mat(group_list(i));
    fprintf('\n');
    fprintf(1,'%s ',['Analyzing ' group]);
    Y_i = X(:,strcmp(groups, group));
    R = size(Y_i,1);
    s = thr_min;
    
    % 2. Generate N bootstrap samples of size M
    graphs = bootstrap_graphs(Y_i,boot_n,boot_m,fdr_thres<1 && ~strcmp(fdr_type,'boots'),fdr_thres);
    
    % Threshold by significance (FDR)
    if (fdr_thres < 1)
         if strcmp(fdr_type,'normal')
             for k = 1 : boot_n
                 %graphs.r_boot(:,:,k) = graphs.r_boot(:,:,k).*double(graphs.p_boot(:,:,k) < fdr_thres);
                 graphs.r_boot(:,:,k) = graphs.r_boot(:,:,k).*double(graphs.fdr_boot(:,:,k));
             end
         elseif strcmp(fdr_type,'boots')
             % Use mean critical P over bootstraps, for all bootstraps
             p_crit=mean(graphs.fdr_crit);
             p_avr=mean(graphs.p_boot,3);
             fdr_mask=double(p_avr<p_crit);
             for k = 1 : boot_n
                graphs.r_boot(:,:,k) = graphs.r_boot(:,:,k).*fdr_mask;
             end
         end
    end
    
    % Remove negative values
    if remove_neg
        for k = 1 : boot_n
             graphs.r_boot(:,:,k) = graphs.r_boot(:,:,k).*double(graphs.r_boot(:,:,k) > 0);
        end
    end
    
    if keep_boots
       results.(group).boots = graphs; 
       results.(group).mean_r = mean(graphs.r_boot,3);
       results.(group).std_r = std(graphs.r_boot,0,3);
    end
    
    j = 1;
    
    results.(group).global_E = zeros(thr_count, boot_n);
    results.(group).local_E = zeros(thr_count, boot_n);
    results.(group).costs = zeros(thr_count,boot_n);
    results.(group).reg_costs = zeros(thr_count,boot_n, R);
    results.(group).reg_global_E = zeros(thr_count, boot_n, R);
    results.(group).reg_local_E = zeros(thr_count, boot_n, R);
    results.(group).deg_centr = zeros(thr_count, boot_n, R);
    results.(group).betw_centr = zeros(thr_count, boot_n, R);
    
    elapsed_g=0;
    while s <= thr_max
        
        tic;
        results.thresholds(j) = s;
        
        r_boot = graphs.r_boot;
        
        global_E = zeros(boot_n, 1);
        local_E = zeros(boot_n, 1);
        reg_global_E = zeros(boot_n, R);
        reg_local_E = zeros(boot_n, R);
        deg_centr = zeros(boot_n, R);
        betw_centr = zeros(boot_n, R);
        
        % 3. For each sample, obtain a binary matrix of the desired sparsity and
        %    compute graph measure of interest
        %    Use 'parfor' for parallel computation
        
        %parfor b = 1 : boot_n
        for b = 1 : boot_n
            if strcmp(threshold_by,'sparsity')
                [b_graph, m_graph] = threshold_by_sparsity(r_boot(:,:,b),s);
                results.(group).costs(j,b)=s;
                %results.(group).min_pearson(j,b)=min(m_graph(m_graph>0));
            elseif strcmp(threshold_by,'pearson')
                b_graph = r_boot(:,:,b) > s;
                results.(group).costs(j,b)=get_sparsity(b_graph);
                %results.(group).min_pearson(j,b)=s;
            elseif strcmp(threshold_by,'abs_pearson')
                b_graph = abs(r_boot(:,:,b)) > s;
                results.(group).costs(j,b)=get_sparsity(b_graph);
                %results.(group).min_pearson(j,b) = s;
            end
            
             % 4. Add result to list
            eff = gretna_efficiency(b_graph);
            global_E(b) = eff.gE;
            local_E(b) = eff.locE;
            reg_global_E(b,:) = eff.regionalglobalE;
            reg_local_E(b,:) = eff.regionallocalE;
            deg_centr(b,:) = sum(b_graph,1);
            betw_centr(b,:) = deg_centr(b,:); % temp: betweenness_wei(b_graph);
            
            
        end
        
        results.(group).global_E(j,:) = global_E;
        results.(group).local_E(j,:) = local_E;
        results.(group).reg_global_E(j,:,:) = reg_global_E;
        results.(group).reg_local_E(j,:,:) = reg_local_E;
        results.(group).deg_centr(j,:,:) = deg_centr;
        results.(group).betw_centr(j,:,:) = betw_centr;
        
        t = toc;
        elapsed_g = elapsed_g + t;
        avr = elapsed_g / j;
        
        fprintf('\n%s', ['Finished threshold ' num2str(s) '; took ' num2str(t) 's; avr ' ...
                         num2str(avr) 's.'] );
        
        j = j + 1;
        s = s + thr_step;
    end
    
    % Normalize centrality measures
    if sum(results.(group).deg_centr(:)) > 0
        % Max connections is R - 1
        results.(group).deg_centr = normalize_data2(results.(group).deg_centr,0,1,0,R-1);
    end 
    if sum(results.(group).betw_centr(:)) > 0
        results.(group).betw_centr = normalize_data(results.(group).betw_centr);
        results.(group).betw_centr(isnan(results.(group).betw_centr)) = 0;
    end 
    
    elapsed = elapsed + elapsed_g;
    ge = datestr(datenum(0,0,0,0,0,elapsed_g),'HH:MM:SS');
    te = datestr(datenum(0,0,0,0,0,elapsed),'HH:MM:SS');
    
    fprintf('\n%s', ['Done; took ' ge '; total elapsed: ' te]);
end
fprintf('\nComputing stats and integrals...');

% Statistical analyses of bootstrap results

    % 3. Group comparisons
    try
        results = corr_net_binary_stats(results, group_list, params);
    catch err
        fprintf('\n%s\n', ['Error: ' err.message] ); 
        rethrow(err);
    end

te = datestr(datenum(0,0,0,0,0,elapsed),'HH:MM:SS');
fprintf('\n%s\n', ['Done. Took ' te '.'] );

% Generate plots if desired
if (show_plots)
    window_size=[];
   if isfield(params, 'window_size')
      window_size = params.window_size; 
   end
   x_label = 'Cost (1-Sparsity)';
   if ~strcmp(threshold_by,'sparsity')
      x_label = 'Pearson coefficient'; 
   end
   try
       plot_params=plots_corr_net_binary_params;
       plot_params.x_label = x_label;
       plot_params.window_size=window_size;
       plots_corr_net_binary2(results, group_list, params);
       plot_sparsity_v_pearson(X',[0:0.01:1],groups,group_list);
   catch err
      fprintf('\n%s\n', ['Error: ' err.message] ); 
      rethrow(err);
   end
end
