function [ results ] = analyze_adni_weighted ( regional_thickness, groups, group_list, params );

% Analyzes regional thickness values by generating bootstrap samples (with
% replacement) and computing graph efficiency measures on each sample. This
% method generates weighted graphs using either the correlation coefficient, 
% r, or the slope, b. Only positive values are used for graph construction.
%
% RETURNS:
%   A result structure containing all bootstrap results.
%
% Andrew Reid
% Montreal Neurological Institute
% 2011

boot_n = 20;
boot_m = 100;
%scale_exp = 1;      % Linear scaling
%hard_threshold = 0.01;
show_plots=true;
keep_boots=false;
remove_neg=true;

if nargin > 3
    boot_n = params.boot_n;
    boot_m = params.boot_m;
    %scale_exp = params.scale_exp;
    %hard_threshold = params.hard_threshold;
    show_plots=params.show_plots;
    keep_boots=params.keep_boots;
    remove_neg=params.remove_neg;
end

if nargin < 4
   show_plots = false; 
end

% Normalize each region's data to the range [0, 1]
regional_norm=normalize_data(regional_thickness,false);

% 1. For each group
for i = 1 : length(group_list);
    
    % For each sample, get graph & efficiency measures
    group = cell2mat(group_list(i));
    fprintf('\n');
    fprintf(1,'%s ',['Analyzing ' group]);
    Y_i = regional_norm(:,strcmp(groups, group));
    N = size(Y_i,1);
    
    % 2. Generate N bootstrap samples of size M
    graphs = bootstrap_graphs(Y_i,boot_n,boot_m, true);
    
    if keep_boots
       results.(group).boots = graphs; 
       results.(group).mean_r = mean(graphs.r_boot,3);
       results.(group).std_r = std(graphs.r_boot,0,3);
    end
    
    sparsity = zeros(boot_n, 1);
    e_global = zeros(boot_n, 1);
    e_local = zeros(boot_n, 1);
    e_global_regional = zeros(boot_n, N);
    e_local_regional = zeros(boot_n, N);
    
    
    % Replace with parfor for parallel computation
    for b = 1 : boot_n
        e_graph=graphs.e_boot(:,:,b);
        
        if remove_neg
            % Only positive correlations
            e_graph=e_graph.*(e_graph > 0);
        end
        %e_graph=normalize_data2(e_graph,-1,1,0,1);
        
        % Get weighted graph, scaled to [0,1] with exponent
        %e_graph=create_weighted_graph(e_graph, scale_exp,'positive', hard_threshold);
        sparsity(b)=get_sparsity(e_graph);
        
        % Get weighted efficiency
        eff = compute_weighted_efficiency(e_graph);
        
        e_global(b) = eff.e_global;
        e_local(b) = eff.e_local;
        e_global_regional(b,:) = eff.e_global_regional;
        e_local_regional(b,:) = eff.e_local_regional;
        
    end
    
    results.(group).sparsity = sparsity;
    results.(group).e_global = e_global;
    results.(group).e_local = e_local;
    results.(group).e_global_regional(:,:) = e_global_regional;
    results.(group).e_local_regional(:,:) = e_local_regional;
   
    fprintf('OK.');
end


if show_plots
    
    plots_adni_weighted(results, group_list);
    
end