function [ results ] = analyze_adni_binary_pcorr ( regional_thickness, groups, group_list, params );

% Analyzes regional thickness values by generating bootstrap samples (with
% replacement) and computing graph efficiency measures on each sample. This
% method generates binary graphs using a range of sparsity values,
% specified by the params argument. Only positive values are used for graph
% construction.
%
% RETURNS:
%   A result structure containing all bootstrap results, as well as mean,
%   standard deviation, and integrals calculated across a sparsity range
%   (this is the entire range unless specified by the params argument).
%
% Andrew Reid
% Montreal Neurological Institute
% 2011

sp_min = 0.05;
sp_max = 0.4;
sp_step = 0.025;
boot_n = 20;
boot_m = 100;
int_start = 0.1;
int_end = 0.3;
integral_type='trap';
remove_neg=true;
show_plots=true;
keep_boots=false;

if nargin > 3
    sp_min = params.sp_min;
    sp_max = params.sp_max;
    sp_step = params.sp_step;
    boot_n = params.boot_n;
    boot_m = params.boot_m;
    int_start = params.int_start;
    int_end = params.int_end;
    integral_type=params.integral_type;
    remove_neg=params.remove_neg;
    show_plots=params.show_plots;
    keep_boots=params.keep_boots;
end

if nargin < 4
   show_plots = false; 
end

if nargin < 6
    keep_boots = false;
end

% Bootstrap with residuals:
% 1. Define a sparsity range + interval


j = 0;
s = sp_min;
int_s = -1;
int_e = -1;
while s <= sp_max
    j = j + 1;
    if s >= int_start && int_s < 0
       int_s = j; 
    end
    if s > int_end && int_e < 0
       int_s = j - 1; 
    end
    s = s + sp_step;
end

sp_count = j;

if int_e < 0
   int_e = sp_count; 
end
    
g = length(group_list);
for i = 1 : g
    % Get values for this group
    group = cell2mat(group_list(i));
    fprintf('\n');
    fprintf(1,'%s ',['Analyzing ' group]);
    Y_i = regional_thickness(:,strcmp(groups, group));
    N = size(Y_i,1);
    results.(group).N=N;
    s = sp_min;
    
    % 2. Generate N bootstrap samples of size M
    graphs = bootstrap_graphs_pcorr(Y_i,boot_n,boot_m, false);
    
    % Threshold by significance
%     for g = 1 : boot_n
%         graphs.r_boot(:,:,g) = graphs.r_boot(:,:,g).*double(graphs.p_boot(:,:,g) < 0.05);
%         graphs.r_boot(:,:,g) = graphs.r_boot(:,:,g).*double(graphs.fdr_boot(:,:,g));
%     end
    
    % Remove negative values
    if remove_neg
        for g = 1 : boot_n
             graphs.r_boot(:,:,g) = graphs.r_boot(:,:,g).*double(graphs.r_boot(:,:,g) > 0);
        end
    end
    
    if keep_boots
       results.(group).boots = graphs; 
       results.(group).mean_r = mean(graphs.r_boot,3);
       results.(group).std_r = std(graphs.r_boot,0,3);
    end
    
    j = 1;
    
    results.(group).global_E = zeros(sp_count, boot_n);
    results.(group).local_E = zeros(sp_count, boot_n);
    results.(group).reg_global_E = zeros(sp_count, boot_n, N);
    results.(group).reg_local_E = zeros(sp_count, boot_n, N);
    
    while s <= sp_max
        
        results.sparsities(j) = s;
        
        r_boot = graphs.r_boot;
        
        global_E = zeros(boot_n, 1);
        local_E = zeros(boot_n, 1);
        reg_global_E = zeros(boot_n, N);
        reg_local_E = zeros(boot_n, N);
        mean_sparsity = 0;
        
        % 3. For each sample, obtain a binary matrix of the desired sparsity and
        %    compute graph measure of interest
        %    Use 'parfor' for parallel computation
        
        %parfor b = 1 : boot_n
        for b = 1 : boot_n
            b_graph = threshold_by_sparsity(r_boot(:,:,b),s);
            mean_sparsity = mean_sparsity + get_sparsity(b_graph);
            %test = get_sparsity(b_graph);
            
             % 4. Add result to list
            eff = gretna_efficiency(b_graph);
            global_E(b) = eff.gE;
            local_E(b) = eff.locE;
            reg_global_E(b,:) = eff.regionalglobalE;
            reg_local_E(b,:) = eff.regionallocalE;
        end
        
        mean_sparsity = mean_sparsity / boot_n;
        results.(group).mean_sparsity(j) = mean_sparsity;
        results.(group).global_E(j,:) = global_E;
        results.(group).local_E(j,:) = local_E;
        results.(group).reg_global_E(j,:,:) = reg_global_E;
        results.(group).reg_local_E(j,:,:) = reg_local_E;
        
        j = j + 1;
        s = s + sp_step;
        fprintf('.');
    end
    
    fprintf('OK.');
end
fprintf('\nComputing stats and integrals...\n');

% Statistical analyses of bootstrap results



    % 3. Group comparisons
results = adni_binary_stats(results, group_list, params);

% i = 1;
% for g1 = 1 : length(group_list);
%     group1 = cell2mat(group_list(g1));
%     for g2 = 1 : length(group_list);
%         group2 = cell2mat(group_list(g2));
%         if g1 < g2
%             [h,p,~,stts]=ttest2(results.(group1).integral_global, results.(group2).integral_global);
%             results.stats.integral_global.t(i)=stts.tstat;
%             results.stats.integral_global.p(i)=p;
%             [h,p,~,stts]=ttest2(results.(group1).integral_local, results.(group2).integral_local);
%             results.stats.integral_local.t(i)=stts.tstat;
%             results.stats.integral_local.p(i)=p;
%             i = i + 1;
%         end
%     end
%     
% end
% 
%  % Bonferroni correction
%  alpha = 0.05/(i - 1);
%  results.stats.integral_global.sig=results.stats.integral_global.p < alpha;
%  results.stats.integral_local.sig=results.stats.integral_local.p < alpha;
    
fprintf('Done.\n');

% Generate plots if desired
if (show_plots)
    
   plots_adni_binary(results, group_list);

end
