%% Analysis for ADNI Dataset - Pearson

% Initiate
% Run this for all sections
adni_dir = 'C:\Users\Andrew\Documents\MATLAB\adni_areid';
output_dir = 'C:\Users\Andrew\Documents\Projects\ADNI\matlab\pearson';
pajek_dir = fullfile(output_dir, 'pajek');

if ~exist(pajek_dir,'dir')
    mkdir(pajek_dir);
end

%% 1. CT, Residuals after Age, Sex

%% 1.1. All subjects

% Load data
load ( fullfile (adni_dir, 'final_data', 'ct_civet_resid_1665_15t.mat') );
load ( fullfile (adni_dir, 'final_data', 'aal_on_minc') );

% Get regional means
rm_ct_civet = get_regional_means(ct_civet_resid_age_sex_rid, aal_mod_phg, aal_list_76);
clear ('ct_civet_resid_age_sex_rid');
N = size(rm_ct_civet,1);

% Get pearson correlations
[R_ct, P_ct] = get_correlation_matrix2(rm_ct_civet, 0.05, false);

% FDR correction
[~, crit_p] = fdr_bky ( P_ct);
R_ct_fdr = R_ct.*(P_ct < crit_p);

% Only positive
R_ct_fdr = R_ct_fdr.*double(R_ct_fdr>0);

% Visualize
clear matrices;
matrices(1,:,:) = R_ct;
matrices(2,:,:) = R_ct_fdr;
plot_params = plot_matrices_params(matrices, [{'Pearson Correlation - CT (uncorrected)'}, ...
                                              {'Pearson Correlation - CT (FDR corrected)'}]);

plot_params(1).show_cbar=true;      
plot_params(2).show_cbar=true;    
plot_matrices(plot_params,1,2);

%% 1.1.1. Export to Pajek
pajek_dir = fullfile(output_dir, 'pajek');
write_matrix_to_pajek(R_ct,fullfile(pajek_dir, 'ct_raw_corr.net'), 'directed', false, ... 
                      'weighted', true, 'coords', aal_centroids_76, 'labels', aal_names_76);
write_matrix_to_pajek(R_ct_fdr,fullfile(pajek_dir, 'ct_raw_corr_fdr.net'), 'directed', false, ... 
                      'weighted', true, 'coords', aal_centroids_76, 'labels', aal_names_76);

%% 1.1.2. Centrality measures

% Degree centrality
dc_ct_fdr = sum(R_ct_fdr,1);
dc_ct_fdr = normalize_data2(dc_ct_fdr);
write_matrix(fullfile(pajek_dir, 'dc_ct_fdr.txt'), dc_ct_fdr', '\t');

% Betweenness centrality
%T=sqrt(1-R_ct_fdr.^2);
T=1-R_ct_fdr;
bc_ct_fdr = betweenness_wei(T);
bc_ct_fdr = normalize_data2(bc_ct_fdr);
write_matrix(fullfile(pajek_dir, 'bc_ct_fdr.txt'), bc_ct_fdr, '\t');
           
              
%% 1.2. By clinical groupings

% Load subject data
load ( fullfile(adni_dir, 'final_data', 'subject_data_validated_15t.mat') );
groups = subject_data(:,4);

rm_ct_nc = rm_ct_civet(strcmp(groups,'Normal'),:);
rm_ct_mci = rm_ct_civet(strcmp(groups,'MCI'),:);
rm_ct_ad = rm_ct_civet(strcmp(groups,'AD'),:);

% Get pearson correlations
[R_ct_nc, P_ct_nc] = get_correlation_matrix2(rm_ct_nc, 0.05, false);
[R_ct_mci, P_ct_mci] = get_correlation_matrix2(rm_ct_mci, 0.05, false);
[R_ct_ad, P_ct_ad] = get_correlation_matrix2(rm_ct_ad, 0.05, false);

% Visualize uncorrected
matrices(1,:,:) = R_ct_nc;
matrices(2,:,:) = R_ct_mci;
matrices(3,:,:) = R_ct_ad;
plot_params = plot_matrices_params(matrices, [{'Normal'}, ...
                                              {'MCI'}, ...
                                              {'AD'}], true );

plot_options.show_cbar=true;
plot_options.plot_title='Pearson Correlations by Group (uncorrected)';
plot_options.plot_size=[950,450];

plot_matrices(plot_params,1,3, plot_options);

% FDR correction
[~, crit_p] = fdr_bky ( P_ct_nc);
R_ct_nc_fdr = R_ct_nc.*(P_ct_nc < crit_p);
[~, crit_p] = fdr_bky ( P_ct_mci);
R_ct_mci_fdr = R_ct_mci.*(P_ct_mci < crit_p);
[~, crit_p] = fdr_bky ( P_ct_ad);
R_ct_ad_fdr = R_ct_ad.*(P_ct_ad < crit_p);

% Visualize corrected
matrices(1,:,:) = R_ct_nc_fdr;
matrices(2,:,:) = R_ct_mci_fdr;
matrices(3,:,:) = R_ct_ad_fdr;
plot_params = plot_matrices_params(matrices, [{'Normal'}, ...
                                              {'MCI'}, ...
                                              {'AD'}], ...
                                              true, [0,1]);

plot_options.show_cbar=true;
plot_options.plot_title='Pearson Correlation by Group (FDR-corrected)';
plot_options.plot_size=[950,450];

plot_matrices(plot_params,1,3, plot_options);

% Write to Pajek
write_matrix_to_pajek(R_ct_nc_fdr,fullfile(pajek_dir, 'ct_raw_corr_nc_fdr.net'), 'directed', false, ... 
                      'weighted', true, 'coords', aal_centroids_76, 'labels', aal_names_76);
write_matrix_to_pajek(R_ct_mci_fdr,fullfile(pajek_dir, 'ct_raw_corr_mci_fdr.net'), 'directed', false, ... 
                      'weighted', true, 'coords', aal_centroids_76, 'labels', aal_names_76);
write_matrix_to_pajek(R_ct_ad_fdr,fullfile(pajek_dir, 'ct_raw_corr_ad_fdr.net'), 'directed', false, ... 
                      'weighted', true, 'coords', aal_centroids_76, 'labels', aal_names_76);


% Output sparsity after FDR
s=get_sparsity(R_ct_nc_fdr);
fprintf('\n%s: %f\n', 'Sparsity for Normal Control group (after FDR): ', s);
s=get_sparsity(R_ct_mci_fdr);
fprintf('\n%s: %f\n', 'Sparsity for MCI group (after FDR): ', s);
s=get_sparsity(R_ct_ad_fdr);
fprintf('\n%s: %f\n', 'Sparsity for AD group (after FDR): ', s);

% Positive only
R_ct_nc_fdr=R_ct_nc_fdr.*(R_ct_nc_fdr>0);
R_ct_mci_fdr=R_ct_mci_fdr.*(R_ct_mci_fdr>0);
R_ct_ad_fdr=R_ct_ad_fdr.*(R_ct_ad_fdr>0);

% Network measures
pajek_dir = fullfile(output_dir, 'pajek');

% Degree centrality
dc_ct_nc_fdr=sum(R_ct_nc_fdr);
dc_ct_mci_fdr=sum(R_ct_mci_fdr);
dc_ct_ad_fdr=sum(R_ct_ad_fdr);
dc_ct_nc_fdr=normalize_data2(dc_ct_nc_fdr);
dc_ct_mci_fdr=normalize_data2(dc_ct_mci_fdr);
dc_ct_ad_fdr=normalize_data2(dc_ct_ad_fdr);

write_matrix(fullfile(pajek_dir, 'dc_ct_nc_fdr.txt'), dc_ct_nc_fdr', '\t');
write_matrix(fullfile(pajek_dir, 'dc_ct_mci_fdr.txt'), dc_ct_mci_fdr', '\t');
write_matrix(fullfile(pajek_dir, 'dc_ct_ad_fdr.txt'), dc_ct_ad_fdr', '\t');

% Betweenness centrality
bc_ct_nc_fdr=betweenness_wei(1-R_ct_nc_fdr);
bc_ct_mci_fdr=betweenness_wei(1-R_ct_mci_fdr);
bc_ct_ad_fdr=betweenness_wei(1-R_ct_ad_fdr);

write_matrix(fullfile(pajek_dir, 'bc_ct_nc_fdr.txt'), bc_ct_nc_fdr, '\t');
write_matrix(fullfile(pajek_dir, 'bc_ct_mci_fdr.txt'), bc_ct_mci_fdr, '\t');
write_matrix(fullfile(pajek_dir, 'bc_ct_ad_fdr.txt'), bc_ct_ad_fdr, '\t');


%% 1.2.1. Group comparisons

% Normal vs. MCI
[ D_nc_mci, P ] = z_comparison (R_ct_nc, R_ct_mci, N);
[~, crit_p] = fdr_bky ( P );
D_nc_mci = D_nc_mci.*( P < crit_p );
D_nc_mci = D_nc_mci.*( R_ct_nc_fdr ~= 0);
D_nc_mci = D_nc_mci.*( R_ct_mci_fdr ~= 0);

% MCI vs. AD
[ D_mci_ad, P ] = z_comparison (R_ct_mci, R_ct_ad, N);
[~, crit_p] = fdr_bky ( P );
D_mci_ad = D_mci_ad.*( P < crit_p );
D_mci_ad = D_mci_ad.*( R_ct_ad_fdr ~= 0);
D_mci_ad = D_mci_ad.*( R_ct_mci_fdr ~= 0);

% Normal vs. AD
[ D_nc_ad, P ] = z_comparison (R_ct_nc, R_ct_ad, N);
[~, crit_p] = fdr_bky ( P );
D_nc_ad = D_nc_ad.*( P < crit_p );
D_nc_ad = D_nc_ad.*( R_ct_ad_fdr ~= 0);
D_nc_ad = D_nc_ad.*( R_ct_nc_fdr ~= 0);

% Visualize differences
matrices(1,:,:) = D_nc_mci;
matrices(2,:,:) = D_mci_ad;
matrices(3,:,:) = D_nc_ad;
plot_params = plot_matrices_params(matrices, [{'Normal - MCI'}, ...
                                              {'MCI - AD'}, ...
                                              {'Normal - AD'}], ...
                                              true, [-7, 7]);
plot_options.show_cbar=true;
plot_options.plot_title='Pearson Z-Comparisons (FDR-corrected)';
plot_options.plot_size=[950,450];

plot_matrices(plot_params,1,3, plot_options);

% Export to file
B = size(D_nc_mci,1);
dc_z_nc_mci_pos = sum(D_nc_mci.*double(D_nc_mci>0)) / B;
dc_z_nc_mci_neg = sum(D_nc_mci.*double(D_nc_mci<0))/ B;
dc_z_mci_ad_pos = sum(D_nc_mci.*double(D_mci_ad>0))/ B;
dc_z_mci_ad_neg = sum(D_nc_mci.*double(D_mci_ad<0))/ B;
dc_z_nc_ad_pos = sum(D_nc_mci.*double(D_nc_ad>0))/ B;
dc_z_nc_ad_neg = sum(D_nc_mci.*double(D_nc_ad<0))/ B;


write_matrix(fullfile(pajek_dir, 'dc_z_nc_mci_pos.txt'), dc_z_nc_mci_pos', '\t');
write_matrix(fullfile(pajek_dir, 'dc_z_nc_mci_neg.txt'), dc_z_nc_mci_neg', '\t');
write_matrix(fullfile(pajek_dir, 'dc_z_mci_ad_pos.txt'), dc_z_mci_ad_pos', '\t');
write_matrix(fullfile(pajek_dir, 'dc_z_mci_ad_neg.txt'), dc_z_mci_ad_neg', '\t');
write_matrix(fullfile(pajek_dir, 'dc_z_nc_ad_pos.txt'), dc_z_nc_ad_pos', '\t');
write_matrix(fullfile(pajek_dir, 'dc_z_nc_ad_neg.txt'), dc_z_nc_ad_neg', '\t');


%% 1.2.2. Basic Network Measures

% Degree centrality
dc_ct_nc_fdr = sum(R_ct_nc_fdr,1);
dc_ct_nc_fdr = normalize_data2(dc_ct_nc_fdr);

dc_ct_mci_fdr = sum(R_ct_mci_fdr,1);
dc_ct_mci_fdr = normalize_data2(dc_ct_mci_fdr);

dc_ct_ad_fdr = sum(R_ct_ad_fdr,1);
dc_ct_ad_fdr = normalize_data2(dc_ct_ad_fdr);

% Comparisons (requires bootstrapping)
 
% Uncomment to obtain bootstraps; otherwise load precomputed samples from file
%   tic
% % % boots_ct_100 = bootstrap_graphs(rm_ct_civet_std',100,100,false,1,'partial',true);
%   boots_ct_nc_100 = bootstrap_graphs(rm_ct_nc',100,400,false,0.05,'pearson',true);
%   boots_ct_mci_100 = bootstrap_graphs(rm_ct_mci',100,400,false,0.05,'pearson',true);
%   boots_ct_ad_100 = bootstrap_graphs(rm_ct_ad',100,400,false,0.05,'pearson',true);
%   toc
load ( fullfile(adni_dir, 'final_data', 'ct_civet_boots.mat') );

B = size(boots_ct_nc_100.r_boot,3);

% Apply FDR threshold 
boots_ct_nc_100.r_boot=boots_ct_nc_100.r_boot.*double(boots_ct_nc_100.fdr_boot);
boots_ct_mci_100.r_boot=boots_ct_mci_100.r_boot.*double(boots_ct_mci_100.fdr_boot);
boots_ct_ad_100.r_boot=boots_ct_ad_100.r_boot.*double(boots_ct_ad_100.fdr_boot);

% Take only positive values
boots_ct_nc_100.r_boot=boots_ct_nc_100.r_boot.*double(boots_ct_nc_100.r_boot>0);
boots_ct_mci_100.r_boot=boots_ct_mci_100.r_boot.*double(boots_ct_mci_100.r_boot>0);
boots_ct_ad_100.r_boot=boots_ct_ad_100.r_boot.*double(boots_ct_ad_100.r_boot>0);

% Mean, st. dev
boots_ct_nc_100.mean_r=mean(boots_ct_nc_100.r_boot,3);
boots_ct_nc_100.stdev_r=std(boots_ct_nc_100.r_boot,0,3);
boots_ct_mci_100.mean_r=mean(boots_ct_mci_100.r_boot,3);
boots_ct_mci_100.stdev_r=std(boots_ct_mci_100.r_boot,0,3);
boots_ct_ad_100.mean_r=mean(boots_ct_ad_100.r_boot,3);
boots_ct_ad_100.stdev_r=std(boots_ct_ad_100.r_boot,0,3);

% Get DC
for i = 1 : B
    boots_ct_nc_100.dc(i,:) = sum(boots_ct_nc_100.r_boot(:,:,i));
    boots_ct_mci_100.dc(i,:) = sum(boots_ct_mci_100.r_boot(:,:,i));
    boots_ct_ad_100.dc(i,:) = sum(boots_ct_ad_100.r_boot(:,:,i));
end

boots_ct_nc_100.dc = normalize_data2(boots_ct_nc_100.dc);
boots_ct_mci_100.dc = normalize_data2(boots_ct_mci_100.dc);
boots_ct_ad_100.dc = normalize_data2(boots_ct_ad_100.dc);
boots_ct_nc_100.dc(isnan(boots_ct_nc_100.dc))=0;
boots_ct_mci_100.dc(isnan(boots_ct_mci_100.dc))=0;
boots_ct_ad_100.dc(isnan(boots_ct_ad_100.dc))=0;

% Get mean/std
dc_ct_nc_boot_mean = mean(boots_ct_nc_100.dc,1);
dc_ct_mci_boot_mean = mean(boots_ct_mci_100.dc,1);
dc_ct_ad_boot_mean = mean(boots_ct_ad_100.dc,1);
dc_ct_nc_boot_stdev = std(boots_ct_nc_100.dc,0,1);
dc_ct_mci_boot_stdev = std(boots_ct_mci_100.dc,0,1);
dc_ct_ad_boot_stdev = std(boots_ct_ad_100.dc,0,1);

% Write to file
write_matrix(fullfile(pajek_dir, 'dc_mean_boot_nc_fdr.txt'), dc_ct_nc_boot_mean', '\t');
write_matrix(fullfile(pajek_dir, 'dc_mean_boot_mci_fdr.txt'), dc_ct_mci_boot_mean', '\t');
write_matrix(fullfile(pajek_dir, 'dc_mean_boot_ad_fdr.txt'), dc_ct_ad_boot_mean', '\t');


% Get t-values
N2 = length(boots_ct_nc_100.mean_r);
sig_labels = repmat('~',N2,3);
[~,p,~,stats] = ttest2(boots_ct_nc_100.dc,boots_ct_mci_100.dc);
t_dc_ct_nc_mci = stats.tstat;
fdr_dc_ct_nc_mci = fdr_bky(p);
sig_t_dc_ct_nc_mci = t_dc_ct_nc_mci(p < 0.05 & fdr_dc_ct_nc_mci);
t_dc_ct_nc_mci_fdr=t_dc_ct_nc_mci.*double(p < 0.05 & fdr_dc_ct_nc_mci);
sig_labels(p < 0.05 & fdr_dc_ct_nc_mci,1)='a';

[~,p,~,stats] = ttest2(boots_ct_mci_100.dc,boots_ct_ad_100.dc);
t_dc_ct_mci_ad = stats.tstat;
fdr_dc_ct_mci_ad = fdr_bky(p);
sig_t_dc_ct_mci_ad = t_dc_ct_mci_ad(p < 0.05 & fdr_dc_ct_mci_ad);
t_dc_ct_mci_ad_fdr=t_dc_ct_mci_ad.*double(p < 0.05 & fdr_dc_ct_mci_ad);
sig_labels(p < 0.05 & fdr_dc_ct_mci_ad,2)='b';

[~,p,~,stats] = ttest2(boots_ct_nc_100.dc,boots_ct_ad_100.dc);
t_dc_ct_nc_ad = stats.tstat;
fdr_dc_ct_nc_ad = fdr_bky(p);
sig_t_dc_ct_nc_ad = t_dc_ct_nc_ad(p < 0.05 & fdr_dc_ct_nc_ad);
t_dc_ct_nc_ad_fdr=t_dc_ct_nc_ad.*double(p < 0.05 & fdr_dc_ct_nc_ad);
sig_labels(p < 0.05 & fdr_dc_ct_nc_ad,3)='c';


% Plot as bar graphs (grouped)
M=[dc_ct_nc_boot_mean; dc_ct_mci_boot_mean; dc_ct_ad_boot_mean]';
denom=sqrt(B);
M_err=[dc_ct_nc_boot_stdev/denom; dc_ct_mci_boot_stdev/denom; dc_ct_ad_boot_stdev/denom]';

params=plot_horizontal_sorted_params;
params.y_labels=aal_names_76;
params.sort_order='descend';
params.plot_type='grouped';
params.plot_title='Degree centrality, sorted by NC';
params.legend_labels=[{'NC'},{'MCI'},{'AD'}];
params.max_bars=15;
params.limits=[0,1];
params.plot_size=[800 650];
plot_horizontal_sorted(M,params,M_err)
params2=params;
params2.sort_group=2;
params2.plot_title='Degree centrality, sorted by MCI';
plot_horizontal_sorted(M,params2,M_err);
params3=params;
params3.sort_group=3;
params3.plot_title='Degree centrality, sorted by AD';
plot_horizontal_sorted(M,params3,M_err);

% Plot as bar graphs (stacked)
params=plot_horizontal_sorted_params;
params.y_labels=aal_names;
params.sort_order='descend';
params.plot_type='stacked';
params.plot_title='Degree centrality, sorted by NC';
params.legend_labels=[{'NC'},{'MCI'},{'AD'}];
params.max_bars=25;
params.limits=[0,3];
params.plot_size=[800 650];
plot_horizontal_sorted(M,params)
params2=params;
params2.sort_group=2;
params2.plot_title='Degree centrality, sorted by MCI';
plot_horizontal_sorted(M,params2);
params3=params;
params3.sort_group=3;
params3.plot_title='Degree centrality, sorted by AD';
plot_horizontal_sorted(M,params3);



% Betweenness centrality
bc_ct_nc_fdr = betweenness_wei(1-R_ct_nc_fdr);
bc_ct_nc_fdr = normalize_data2(bc_ct_nc_fdr);

bc_ct_mci_fdr = betweenness_wei(1-R_ct_mci_fdr);
bc_ct_mci_fdr = normalize_data2(bc_ct_mci_fdr);

bc_ct_ad_fdr = betweenness_wei(1-R_ct_ad_fdr);
bc_ct_ad_fdr = normalize_data2(bc_ct_ad_fdr);

% Comparisons

for i = 1 : B
    T = betweenness_wei(1-boots_ct_nc_100.r_boot(:,:,i));
    T(isnan(T))=0;
    T = normalize_data2(T);
    boots_ct_nc_100.bc(i,:) = T;
    T = betweenness_wei(1-boots_ct_mci_100.r_boot(:,:,i));
    T(isnan(T))=0;
    T = normalize_data2(T);
    boots_ct_mci_100.bc(i,:) = T;
    T = betweenness_wei(1-boots_ct_ad_100.r_boot(:,:,i));
    T(isnan(T))=0;
    T = normalize_data2(T);
    boots_ct_ad_100.bc(i,:) = T;
end

% Get mean/std
bc_ct_nc_boot_mean = mean(boots_ct_nc_100.bc,1);
bc_ct_mci_boot_mean = mean(boots_ct_mci_100.bc,1);
bc_ct_ad_boot_mean = mean(boots_ct_ad_100.bc,1);
bc_ct_nc_boot_stdev = std(boots_ct_nc_100.bc,0,1);
bc_ct_mci_boot_stdev = std(boots_ct_mci_100.bc,0,1);
bc_ct_ad_boot_stdev = std(boots_ct_ad_100.bc,0,1);

% Write to file
write_matrix(fullfile(pajek_dir, 'bc_mean_boot_nc_fdr.txt'), bc_ct_nc_boot_mean', '\t');
write_matrix(fullfile(pajek_dir, 'bc_mean_boot_mci_fdr.txt'), bc_ct_mci_boot_mean', '\t');
write_matrix(fullfile(pajek_dir, 'bc_mean_boot_ad_fdr.txt'), bc_ct_ad_boot_mean', '\t');

    
% Get t-values
[~,p,~,stats] = ttest2(boots_ct_nc_100.bc,boots_ct_mci_100.bc);
t_bc_ct_nc_mci = stats.tstat;
t_bc_ct_nc_mci(isnan(t_bc_ct_nc_mci))=0;
fdr_bc_ct_nc_mci = fdr_bky(p);
t_bc_ct_nc_mci_fdr=t_bc_ct_nc_mci.*double(fdr_bc_ct_nc_mci & p < 0.05);

[~,p,~,stats] = ttest2(boots_ct_mci_100.bc,boots_ct_ad_100.bc);
t_bc_ct_mci_ad = stats.tstat;
t_bc_ct_mci_ad(isnan(t_bc_ct_mci_ad))=0;
fdr_bc_ct_mci_ad = fdr_bky(p);
t_bc_ct_mci_ad_fdr = t_bc_ct_mci_ad.*double(fdr_bc_ct_mci_ad & p < 0.05);

[~,p,~,stats] = ttest2(boots_ct_nc_100.bc,boots_ct_ad_100.bc);
t_bc_ct_nc_ad = stats.tstat;
fdr_bc_ct_nc_ad = fdr_bky(p);
t_bc_ct_nc_ad(isnan(t_bc_ct_nc_ad))=0;
t_bc_ct_nc_ad_fdr=t_bc_ct_nc_ad.*double(fdr_bc_ct_nc_ad & p < 0.05);

% Plot as bar graphs (grouped)
M=[bc_ct_nc_boot_mean; bc_ct_mci_boot_mean; bc_ct_ad_boot_mean]';
denom=sqrt(B);
M_err=[bc_ct_nc_boot_stdev/denom; bc_ct_mci_boot_stdev/denom; bc_ct_ad_boot_stdev/denom]';

params=plot_horizontal_sorted_params;
params.y_labels=aal_names_76;
params.sort_order='descend';
params.plot_type='grouped';
params.plot_title='Betweenness centrality, sorted by NC';
params.legend_labels=[{'NC'},{'MCI'},{'AD'}];
params.max_bars=15;
params.limits=[0,1];
params.plot_size=[800 650];
plot_horizontal_sorted(M,params,M_err)
params2=params;
params2.sort_group=2;
params2.plot_title='Betweenness centrality, sorted by MCI';
plot_horizontal_sorted(M,params2,M_err);
params3=params;
params3.sort_group=3;
params3.plot_title='Betweenness centrality, sorted by AD';
plot_horizontal_sorted(M,params3,M_err);

% Write to file

% Degree centrality
write_matrix(fullfile(pajek_dir, 't_dc_ct_nc_mci_fdr.txt'), t_dc_ct_nc_mci_fdr', '\t');
write_matrix(fullfile(pajek_dir, 't_dc_ct_mci_ad_fdr.txt'), t_dc_ct_mci_ad_fdr', '\t');
write_matrix(fullfile(pajek_dir, 't_dc_ct_nc_ad_fdr.txt'), t_dc_ct_nc_ad_fdr', '\t');
% Absolute values
write_matrix(fullfile(pajek_dir, 'abs_t_dc_ct_nc_mci_fdr.txt'), abs(t_dc_ct_nc_mci_fdr'), '\t');
write_matrix(fullfile(pajek_dir, 'abs_t_dc_ct_mci_ad_fdr.txt'), abs(t_dc_ct_mci_ad_fdr'), '\t');
write_matrix(fullfile(pajek_dir, 'abs_t_dc_ct_nc_ad_fdr.txt'), abs(t_dc_ct_nc_ad_fdr'), '\t');


% Betweenness centrality
write_matrix(fullfile(pajek_dir, 't_bc_ct_nc_mci_fdr.txt'), t_bc_ct_nc_mci_fdr', '\t');
write_matrix(fullfile(pajek_dir, 't_bc_ct_mci_ad_fdr.txt'), t_bc_ct_mci_ad_fdr', '\t');
write_matrix(fullfile(pajek_dir, 't_bc_ct_nc_ad_fdr.txt'), t_bc_ct_nc_ad_fdr', '\t');
% Absolute values
write_matrix(fullfile(pajek_dir, 'abs_t_bc_ct_nc_mci_fdr.txt'), abs(t_bc_ct_nc_mci_fdr'), '\t');
write_matrix(fullfile(pajek_dir, 'abs_t_bc_ct_mci_ad_fdr.txt'), abs(t_bc_ct_mci_ad_fdr'), '\t');
write_matrix(fullfile(pajek_dir, 'abs_t_bc_ct_nc_ad_fdr.txt'), abs(t_bc_ct_nc_ad_fdr'), '\t');


%% 2. Efficiency Analyses

clear;
adni_dir = 'C:\Users\Andrew\Documents\MATLAB\adni_areid';
output_dir = 'C:\Users\Andrew\Documents\Projects\ADNI\matlab\pearson';


%% 2.1. Raw CT: Pearson correlations

% Load data
load ( fullfile (adni_dir, 'final_data', 'ct_civet_resid_1665_15t.mat') );
load ( fullfile (adni_dir, 'final_data', 'aal_on_minc') );

% Get regional means
rm_ct_civet = get_regional_means(ct_civet_resid_age_sex_rid, aal_mod_phg, aal_list_76);
clear ('ct_civet_resid_age_sex_rid');
N = size(rm_ct_civet,1);

% Load subject data
load ( fullfile(adni_dir, 'final_data', 'subject_data_validated_15t.mat') );
groups = subject_data(:,4);

% Run bootstrap analysis
params = analyze_corr_net_binary_params;

params.thr_min=0.1;
params.thr_max=0.6;
params.thr_step=0.1;
params.int_start=0.1;
params.int_end=0.6;
params.boot_n=20;
params.boot_m=300;
params.window_size=[1000,900]

results_raw_ct_boots = analyze_corr_net_binary(rm_ct_civet',groups,[{'Normal'},{'MCI'},{'AD'}],params);



% Export regional efficiency as graph data



% Export centrality as graph data

dc_intgr_nc = results_raw_ct_boots.Normal.mean.integral_deg_centr;
write_matrix(fullfile(pajek_dir, 'dc_intgr_nc.txt'), dc_intgr_nc', '\t');
dc_intgr_mci = results_raw_ct_boots.MCI.mean.integral_deg_centr;
write_matrix(fullfile(pajek_dir, 'dc_intgr_mci.txt'), dc_intgr_mci', '\t');
dc_intgr_ad = results_raw_ct_boots.AD.mean.integral_deg_centr;
write_matrix(fullfile(pajek_dir, 'dc_intgr_ad.txt'), dc_intgr_ad', '\t');
bc_intgr_nc = results_raw_ct_boots.Normal.mean.integral_betw_centr;
write_matrix(fullfile(pajek_dir, 'bc_intgr_nc.txt'), bc_intgr_nc', '\t');
bc_intgr_mci = results_raw_ct_boots.MCI.mean.integral_betw_centr;
write_matrix(fullfile(pajek_dir, 'bc_intgr_mci.txt'), bc_intgr_mci', '\t');
bc_intgr_ad = results_raw_ct_boots.AD.mean.integral_betw_centr;
write_matrix(fullfile(pajek_dir, 'bc_intgr_ad.txt'), bc_intgr_ad', '\t');
