
%% Analysis for ADNI Dataset

% Initiate
% Run this for all sections
adni_dir = 'C:\Users\Andrew\Documents\MATLAB\adni_areid';
output_dir = 'C:\Users\Andrew\Documents\Projects\ADNI\matlab\partial';

%% 1. Raw CT: Partial correlations

%% 1.1. All subjects

% Load data
load ( fullfile (adni_dir, 'final_data', 'ct_civet_1665_15t.mat') );
load ( fullfile (adni_dir, 'final_data', 'aal_on_minc') );

% Get regional means
rm_ct_civet = get_regional_means(Y_ct_civet, aal_mod_phg, aal_list_76);
clear ('Y_ct_civet');
N = size(rm_ct_civet,1);

% Get partial correlations
[R_ct, P_ct] = partialcorr(rm_ct_civet, 'rows', 'complete');

% FDR correction
[~, crit_p] = fdr_bky ( P_ct);
R_ct_fdr = R_ct.*(P_ct < crit_p);

% Visualize
matrices(1,:,:) = R_ct;
matrices(2,:,:) = R_ct_fdr;
plot_params = plot_matrices_params(matrices, [{'Partial Correlation - CT (uncorrected)'}, ...
                                              {'Partial Correlation - CT (FDR corrected)'}]);

plot_params(1).show_cbar=true;      
plot_params(2).show_cbar=true;    
plot_matrices(plot_params,1,2);

%% 1.2. By clinical groupings

% Load subject data
load ( fullfile(adni_dir, 'final_data', 'subject_data_validated_15t.mat') );
groups = subject_data(:,4);

rm_ct_nc = rm_ct_civet(strcmp(groups,'Normal'),:);
rm_ct_mci = rm_ct_civet(strcmp(groups,'MCI'),:);
rm_ct_ad = rm_ct_civet(strcmp(groups,'AD'),:);

% Get partial correlations
[R_ct_nc, P_ct_nc] = partialcorr(rm_ct_nc, 'rows', 'complete');
[R_ct_mci, P_ct_mci] = partialcorr(rm_ct_mci, 'rows', 'complete');
[R_ct_ad, P_ct_ad] = partialcorr(rm_ct_ad, 'rows', 'complete');

% Visualize uncorrected
matrices(1,:,:) = R_ct_nc;
matrices(2,:,:) = R_ct_mci;
matrices(3,:,:) = R_ct_ad;
plot_params = plot_matrices_params(matrices, [{'Normal'}, ...
                                              {'MCI'}, ...
                                              {'AD'}], true );

plot_options.show_cbar=true;
plot_options.plot_title='Partial Correlations by Group (uncorrected)';
plot_options.plot_size=[950,450];

plot_matrices(plot_params,1,3, plot_options);

% FDR correction
[~, crit_p] = fdr_bky ( P_ct_nc);
R_ct_nc_fdr = R_ct_nc.*(P_ct_nc < crit_p);
[~, crit_p] = fdr_bky ( P_ct_mci);
R_ct_mci_fdr = R_ct_mci.*(P_ct_mci < crit_p);
[~, crit_p] = fdr_bky ( P_ct_ad);
R_ct_ad_fdr = R_ct_ad.*(P_ct_ad < crit_p);

% Visualize corrected
matrices(1,:,:) = R_ct_nc_fdr;
matrices(2,:,:) = R_ct_mci_fdr;
matrices(3,:,:) = R_ct_ad_fdr;
plot_params = plot_matrices_params(matrices, [{'Normal'}, ...
                                              {'MCI'}, ...
                                              {'AD'}], ...
                                              true, [0,0.6]);

plot_options.show_cbar=true;
plot_options.plot_title='Partial Correlation by Group (FDR-corrected)';
plot_options.plot_size=[950,450];

plot_matrices(plot_params,1,3, plot_options);

% Output sparsity after FDR
s=get_sparsity(R_ct_nc_fdr);
fprintf('\n%s: %f\n', 'Sparsity for Normal Control group (after FDR): ', s);
s=get_sparsity(R_ct_mci_fdr);
fprintf('\n%s: %f\n', 'Sparsity for MCI group (after FDR): ', s);
s=get_sparsity(R_ct_ad_fdr);
fprintf('\n%s: %f\n', 'Sparsity for AD group (after FDR): ', s);

%% 1.2.1. Group comparisons

% Normal vs. MCI
[ D_nc_mci, P ] = z_comparison (R_ct_nc, R_ct_mci, N);
[~, crit_p] = fdr_bky ( P );
D_nc_mci = D_nc_mci.*( P < crit_p );
D_nc_mci = D_nc_mci.*( R_ct_nc_fdr ~= 0);
D_nc_mci = D_nc_mci.*( R_ct_mci_fdr ~= 0);

% MCI vs. AD
[ D_mci_ad, P ] = z_comparison (R_ct_mci, R_ct_ad, N);
[~, crit_p] = fdr_bky ( P );
D_mci_ad = D_mci_ad.*( P < crit_p );
D_mci_ad = D_mci_ad.*( R_ct_ad_fdr ~= 0);
D_mci_ad = D_mci_ad.*( R_ct_mci_fdr ~= 0);

% Normal vs. AD
[ D_nc_ad, P ] = z_comparison (R_ct_nc, R_ct_ad, N);
[~, crit_p] = fdr_bky ( P );
D_nc_ad = D_nc_ad.*( P < crit_p );
D_nc_ad = D_nc_ad.*( R_ct_ad_fdr ~= 0);
D_nc_ad = D_nc_ad.*( R_ct_nc_fdr ~= 0);

% Visualize differences
matrices(1,:,:) = D_nc_mci;
matrices(2,:,:) = D_mci_ad;
matrices(3,:,:) = D_nc_ad;
plot_params = plot_matrices_params(matrices, [{'Normal - MCI'}, ...
                                              {'MCI - AD'}, ...
                                              {'Normal - AD'}], ...
                                              true, [-7, 7]);
plot_options.show_cbar=true;
plot_options.plot_title='Partial Correlation Z-Comparisons (FDR-corrected)';
plot_options.plot_size=[950,450];

plot_matrices(plot_params,1,3, plot_options);

% Export to file
B = size(D_nc_mci,1);
dc_z_nc_mci_pos = sum(D_nc_mci.*double(D_nc_mci>0)) / B;
dc_z_nc_mci_neg = sum(D_nc_mci.*double(D_nc_mci<0))/ B;
dc_z_mci_ad_pos = sum(D_nc_mci.*double(D_mci_ad>0))/ B;
dc_z_mci_ad_neg = sum(D_nc_mci.*double(D_mci_ad<0))/ B;
dc_z_nc_ad_pos = sum(D_nc_mci.*double(D_nc_ad>0))/ B;
dc_z_nc_ad_neg = sum(D_nc_mci.*double(D_nc_ad<0))/ B;


write_matrix(fullfile(pajek_dir, 'dc_z_nc_mci_pos.txt'), dc_z_nc_mci_pos', '\t');
write_matrix(fullfile(pajek_dir, 'dc_z_nc_mci_neg.txt'), dc_z_nc_mci_neg', '\t');
write_matrix(fullfile(pajek_dir, 'dc_z_mci_ad_pos.txt'), dc_z_mci_ad_pos', '\t');
write_matrix(fullfile(pajek_dir, 'dc_z_mci_ad_neg.txt'), dc_z_mci_ad_neg', '\t');
write_matrix(fullfile(pajek_dir, 'dc_z_nc_ad_pos.txt'), dc_z_nc_ad_pos', '\t');
write_matrix(fullfile(pajek_dir, 'dc_z_nc_ad_neg.txt'), dc_z_nc_ad_neg', '\t');


%% 2. Standardized CT: Partial correlations

%% 2.1. All subjects

% Load data
load ( fullfile (adni_dir, 'final_data', 'ct_civet_1665_std_15t.mat') );
load ( fullfile (adni_dir, 'final_data', 'aal_on_minc') );

% Get regional means
rm_ct_civet_std = get_regional_means(Y_ct_civet_std, aal_mod_phg, aal_list_76);
clear ('Y_ct_civet_std');
N = size(rm_ct_civet_std,1);

% Get partial correlations
[R_ct_std, P_ct_std] = partialcorr(rm_ct_civet_std, 'rows', 'complete');

% FDR correction
[~, crit_p] = fdr_bky ( P_ct_std);
R_ct_std_fdr = R_ct_std.*(P_ct_std < crit_p);

% Visualize
clear ('matrices');
matrices(1,:,:) = R_ct_std;
matrices(2,:,:) = R_ct_std_fdr;

plot_params = plot_matrices_params(matrices, [{'Partial Correlation - Std. CT (uncorrected)'}, ...
                                              {'Partial Correlation - Std. CT (FDR corrected)'}]);

plot_options.show_cbar=true;
plot_options.plot_title='Partial Correlations (std, uncorrected)';
plot_options.plot_size=[1100,700];
plot_matrices(plot_params,1,2,plot_options);

%% 2.1.1. Output matrix to Pajek

% Create output directory if necessary
pajek_dir = fullfile(output_dir, 'pajek');
if ~exist(pajek_dir, 'dir')
   mkdir(pajek_dir)
end


% Write matrix
write_matrix_to_pajek(R_ct_std_fdr,fullfile(pajek_dir, 'r_pcor_ct_std_fdr.net'), 'directed', false, ... 
                      'weighted', true, 'coords', aal_centroids_76, 'labels', aal_names_76);

% Degree centrality
dc_ct_std_fdr = sum(R_ct_std_fdr,1);
dc_ct_std_fdr = normalize_data2(dc_ct_std_fdr);
write_matrix(fullfile(pajek_dir, 'dc_pcor_ct_std_fdr.txt'), dc_ct_std_fdr', '\t');

% Betweenness centrality
T=1-R_ct_std_fdr;
bc_ct_std_fdr = betweenness_wei(T);
bc_ct_std_fdr = normalize_data2(bc_ct_std_fdr);
write_matrix(fullfile(pajek_dir, 'bc_pcor_ct_std_fdr.txt'), bc_ct_std_fdr', '\t');

%% 2.2. By clinical groupings

% Load subject data
load ( fullfile(adni_dir, 'final_data', 'subject_data_validated_15t.mat') );
groups = subject_data(:,4);

rm_ct_std_nc = rm_ct_civet_std(strcmp(groups,'Normal'),:);
rm_ct_std_mci = rm_ct_civet_std(strcmp(groups,'MCI'),:);
rm_ct_std_ad = rm_ct_civet_std(strcmp(groups,'AD'),:);

% Get partial correlations
[R_ct_std_nc, P_ct_std_nc] = partialcorr(rm_ct_std_nc, 'rows', 'complete');
[R_ct_std_mci, P_ct_std_mci] = partialcorr(rm_ct_std_mci, 'rows', 'complete');
[R_ct_std_ad, P_ct_std_ad] = partialcorr(rm_ct_std_ad, 'rows', 'complete');

% Visualize uncorrected
clear ('matrices');
matrices(1,:,:) = R_ct_std_nc;
matrices(2,:,:) = R_ct_std_mci;
matrices(3,:,:) = R_ct_std_ad;
plot_params = plot_matrices_params(matrices, [{'Normal'}, ...
                                              {'MCI'}, ...
                                              {'AD'}], true );

plot_options.show_cbar=true;
plot_options.plot_title='Partial Correlations by Group (std, uncorrected)';
plot_options.plot_size=[950,450];

plot_matrices(plot_params,1,3, plot_options);

% FDR correction
[~, crit_p] = fdr_bky ( P_ct_std_nc);
R_ct_std_nc_fdr = R_ct_std_nc.*(P_ct_std_nc < crit_p);
[~, crit_p] = fdr_bky ( P_ct_std_mci);
R_ct_std_mci_fdr = R_ct_std_mci.*(P_ct_std_mci < crit_p);
[~, crit_p] = fdr_bky ( P_ct_std_ad);
R_ct_std_ad_fdr = R_ct_std_ad.*(P_ct_std_ad < crit_p);

% Visualize corrected
clear ('matrices');
matrices(1,:,:) = R_ct_std_nc_fdr;
matrices(2,:,:) = R_ct_std_mci_fdr;
matrices(3,:,:) = R_ct_std_ad_fdr;
plot_params = plot_matrices_params(matrices, [{'Normal'}, ...
                                              {'MCI'}, ...
                                              {'AD'}], ...
                                              true, [0,0.6]);

plot_options.show_cbar=true;
plot_options.plot_title='Partial Correlation by Group (std, FDR-corrected)';
plot_options.plot_size=[950,450];

plot_matrices(plot_params,1,3, plot_options);

% Output sparsity after FDR
s=get_sparsity(R_ct_std_nc_fdr);
fprintf('\n%s: %f\n', 'Sparsity for Normal Control group (std, after FDR): ', s);
s=get_sparsity(R_ct_std_mci_fdr);
fprintf('\n%s: %f\n', 'Sparsity for MCI group (std, after FDR): ', s);
s=get_sparsity(R_ct_std_ad_fdr);
fprintf('\n%s: %f\n', 'Sparsity for AD group (std, after FDR): ', s);

%% 2.2.1. Group comparisons

% Normal vs. MCI
[ D_nc_mci_std, P ] = z_comparison (R_ct_std_nc, R_ct_std_mci, N);
[~, crit_p] = fdr_bky ( P );
D_nc_mci_std = D_nc_mci_std.*( P < crit_p );
D_nc_mci_std = D_nc_mci_std.*( R_ct_std_nc_fdr ~= 0);
D_nc_mci_std = D_nc_mci_std.*( R_ct_std_mci_fdr ~= 0);

% MCI vs. AD
[ D_mci_ad_std, P ] = z_comparison (R_ct_std_mci, R_ct_std_ad, N);
[~, crit_p] = fdr_bky ( P );
D_mci_ad_std = D_mci_ad_std.*( P < crit_p );
D_mci_ad_std = D_mci_ad_std.*( R_ct_std_ad_fdr ~= 0);
D_mci_ad_std = D_mci_ad_std.*( R_ct_std_mci_fdr ~= 0);

% Normal vs. AD
[ D_nc_ad_std, P ] = z_comparison (R_ct_std_nc, R_ct_std_ad, N);
[~, crit_p] = fdr_bky ( P );
D_nc_ad_std = D_nc_ad_std.*( P < crit_p );
D_nc_ad_std = D_nc_ad_std.*( R_ct_std_ad_fdr ~= 0);
D_nc_ad_std = D_nc_ad_std.*( R_ct_std_nc_fdr ~= 0);

% Visualize differences
clear ('matrices');
matrices(1,:,:) = D_nc_mci_std;
matrices(2,:,:) = D_mci_ad_std;
matrices(3,:,:) = D_nc_ad_std;
plot_params = plot_matrices_params(matrices, [{'Normal - MCI'}, ...
                                              {'MCI - AD'}, ...
                                              {'Normal - AD'}], ...
                                              true, [-7, 7]);
plot_options.show_cbar=true;
plot_options.plot_title='Partial Correlation Z-Comparisons (std, FDR-corrected)';
plot_options.plot_size=[950,450];

plot_matrices(plot_params,1,3, plot_options);


%% 2.2.2. Correlation group difference with atrophy

% Q: Does z-score of difference correlate with atrophy?
% A: No.

% Normal vs. MCI
dc_zcomp_nc_mci_std_pos = sum(abs(D_nc_mci_std.*(D_nc_mci_std~=0)));
is_nz = dc_zcomp_nc_mci_std_pos ~= 0;
ct_diff_nc_mci_std = mean(rm_ct_std_nc,1)-mean(rm_ct_std_mci,1);
[h,p,ci,stats] = ttest2(rm_ct_std_nc,rm_ct_std_mci,0.05,'both','equal',1);
[~,crit_p] = fdr_bky(p);
ct_diff_nc_mci_std=ct_diff_nc_mci_std(p<crit_p & is_nz);
dc_zcomp_nc_mci_std_pos=dc_zcomp_nc_mci_std_pos(p<crit_p & is_nz);
figure;
set(gcf,'Color','w');

subplot(1,3,1);
scatter(ct_diff_nc_mci_std, dc_zcomp_nc_mci_std_pos);
xlabel('Significant NC-MCI (mm)');
ylabel('Absolute Z-Diff (sum)');
xlim([-.2 .2]);
ylim([0 25]);

% MCI vs. AD
dc_zcomp_mci_ad_std_pos = sum(abs(D_mci_ad_std.*(D_mci_ad_std<0)));
is_nz = dc_zcomp_mci_ad_std_pos ~= 0;
ct_diff_mci_ad_std = mean(rm_ct_std_mci,1)-mean(rm_ct_std_ad,1);
[h,p,ci,stats] = ttest2(rm_ct_std_mci,rm_ct_std_ad,0.05,'both','equal',1);
[~,crit_p] = fdr_bky(p);
ct_diff_mci_ad_std=ct_diff_mci_ad_std(p<crit_p & is_nz);
dc_zcomp_mci_ad_std_pos=dc_zcomp_mci_ad_std_pos(p<crit_p & is_nz);

subplot(1,3,2);
scatter(ct_diff_mci_ad_std, dc_zcomp_mci_ad_std_pos);
xlabel('Significant MCI-AD (mm)');
ylabel('Absolute Z-Diff (sum)');
xlim([-.2 .2]);
ylim([0 25]);

% Normal vs. AD
dc_zcomp_nc_ad_std_pos = sum(abs(D_nc_ad_std.*(D_nc_ad_std<0)));
is_nz = dc_zcomp_nc_ad_std_pos ~= 0;
ct_diff_nc_ad_std = mean(rm_ct_std_nc,1)-mean(rm_ct_std_ad,1);
[h,p,ci,stats] = ttest2(rm_ct_std_nc,rm_ct_std_ad,0.05,'both','equal',1);
[~,crit_p] = fdr_bky(p);
ct_diff_nc_ad_std=ct_diff_nc_ad_std(p<crit_p & is_nz);
dc_zcomp_nc_ad_std_pos=dc_zcomp_nc_ad_std_pos(p<crit_p & is_nz);

subplot(1,3,3);
scatter(ct_diff_nc_ad_std, dc_zcomp_nc_ad_std_pos);
xlabel('Significant NC-AD (mm)');
ylabel('Absolute Z-Diff (sum)');
xlim([-.2 .2]);
ylim([0 25]);

%% 2.2.3. Basic Network Measures

% Degree centrality
dc_ct_std_nc_fdr = sum(R_ct_std_nc_fdr,1);
dc_ct_std_nc_fdr = normalize_data2(dc_ct_std_nc_fdr);

dc_ct_std_mci_fdr = sum(R_ct_std_mci_fdr,1);
dc_ct_std_mci_fdr = normalize_data2(dc_ct_std_mci_fdr);

dc_ct_std_ad_fdr = sum(R_ct_std_ad_fdr,1);
dc_ct_std_ad_fdr = normalize_data2(dc_ct_std_ad_fdr);

% Comparisons (requires bootstrapping)
 
% Uncomment to obtain bootstraps; otherwise load precomputed samples from file
%  tic
% % boots_ct_std_pcorr_100 = bootstrap_graphs(rm_ct_civet_std',100,100,false,1,'partial',true);
%  boots_ct_std_nc_pcorr_100 = bootstrap_graphs(rm_ct_std_nc',100,400,false,0.05,'partial',true);
%  boots_ct_std_mci_pcorr_100 = bootstrap_graphs(rm_ct_std_mci',100,400,false,0.05,'partial',true);
%  boots_ct_std_ad_pcorr_100 = bootstrap_graphs(rm_ct_std_ad',100,400,false,0.05,'partial',true);
%  toc
load ( fullfile(adni_dir, 'final_data', 'ct_civet_std_boots_pcorr.mat') );

B = size(boots_ct_std_nc_pcorr_100.r_boot,3);

% Apply FDR threshold 
%boots_ct_std_nc_pcorr_100.r_boot=boots_ct_std_nc_pcorr_100.r_boot.*double(boots_ct_std_nc_pcorr_100.fdr_boot);
%boots_ct_std_mci_pcorr_100.r_boot=boots_ct_std_mci_pcorr_100.r_boot.*double(boots_ct_std_mci_pcorr_100.fdr_boot);
%boots_ct_std_ad_pcorr_100.r_boot=boots_ct_std_ad_pcorr_100.r_boot.*double(boots_ct_std_ad_pcorr_100.fdr_boot);

% Take only positive values
boots_ct_std_nc_pcorr_100.r_boot=boots_ct_std_nc_pcorr_100.r_boot.*double(boots_ct_std_nc_pcorr_100.r_boot>0);
boots_ct_std_mci_pcorr_100.r_boot=boots_ct_std_mci_pcorr_100.r_boot.*double(boots_ct_std_mci_pcorr_100.r_boot>0);
boots_ct_std_ad_pcorr_100.r_boot=boots_ct_std_ad_pcorr_100.r_boot.*double(boots_ct_std_ad_pcorr_100.r_boot>0);

% Mean, st. dev
boots_ct_std_nc_pcorr_100.mean_r=mean(boots_ct_std_nc_pcorr_100.r_boot,3);
boots_ct_std_nc_pcorr_100.stdev_r=std(boots_ct_std_nc_pcorr_100.r_boot,0,3);
boots_ct_std_mci_pcorr_100.mean_r=mean(boots_ct_std_mci_pcorr_100.r_boot,3);
boots_ct_std_mci_pcorr_100.stdev_r=std(boots_ct_std_mci_pcorr_100.r_boot,0,3);
boots_ct_std_ad_pcorr_100.mean_r=mean(boots_ct_std_ad_pcorr_100.r_boot,3);
boots_ct_std_ad_pcorr_100.stdev_r=std(boots_ct_std_ad_pcorr_100.r_boot,0,3);

% Get DC
for i = 1 : B
    boots_ct_std_nc_pcorr_100.dc(i,:) = sum(boots_ct_std_nc_pcorr_100.r_boot(:,:,i));
    boots_ct_std_mci_pcorr_100.dc(i,:) = sum(boots_ct_std_mci_pcorr_100.r_boot(:,:,i));
    boots_ct_std_ad_pcorr_100.dc(i,:) = sum(boots_ct_std_ad_pcorr_100.r_boot(:,:,i));
end

boots_ct_std_nc_pcorr_100.dc = normalize_data2(boots_ct_std_nc_pcorr_100.dc);
boots_ct_std_mci_pcorr_100.dc = normalize_data2(boots_ct_std_mci_pcorr_100.dc);
boots_ct_std_ad_pcorr_100.dc = normalize_data2(boots_ct_std_ad_pcorr_100.dc);
boots_ct_std_nc_pcorr_100.dc(isnan(boots_ct_std_nc_pcorr_100.dc))=0;
boots_ct_std_mci_pcorr_100.dc(isnan(boots_ct_std_mci_pcorr_100.dc))=0;
boots_ct_std_ad_pcorr_100.dc(isnan(boots_ct_std_ad_pcorr_100.dc))=0;

% Get mean/std
dc_ct_std_nc_boot_mean = mean(boots_ct_std_nc_pcorr_100.dc,1);
dc_ct_std_mci_boot_mean = mean(boots_ct_std_mci_pcorr_100.dc,1);
dc_ct_std_ad_boot_mean = mean(boots_ct_std_ad_pcorr_100.dc,1);
dc_ct_std_nc_boot_stdev = std(boots_ct_std_nc_pcorr_100.dc,0,1);
dc_ct_std_mci_boot_stdev = std(boots_ct_std_mci_pcorr_100.dc,0,1);
dc_ct_std_ad_boot_stdev = std(boots_ct_std_ad_pcorr_100.dc,0,1);

% Get t-values
N2 = length(boots_ct_std_nc_pcorr_100.mean_r);
sig_labels = repmat('~',N2,3);
[~,p,~,stats] = ttest2(boots_ct_std_nc_pcorr_100.dc,boots_ct_std_mci_pcorr_100.dc);
t_dc_ct_std_nc_mci = stats.tstat;
fdr_dc_ct_std_nc_mci = fdr_bky(p);
sig_t_dc_ct_std_nc_mci = t_dc_ct_std_nc_mci(p < 0.05 & fdr_dc_ct_std_nc_mci);
sig_labels(p < 0.05 & fdr_dc_ct_std_nc_mci,1)='a';

[~,p,~,stats] = ttest2(boots_ct_std_mci_pcorr_100.dc,boots_ct_std_ad_pcorr_100.dc);
t_dc_ct_std_mci_ad = stats.tstat;
fdr_dc_ct_std_mci_ad = fdr_bky(p);
sig_t_dc_ct_std_mci_ad = t_dc_ct_std_mci_ad(p < 0.05 & fdr_dc_ct_std_mci_ad);
sig_labels(p < 0.05 & fdr_dc_ct_std_mci_ad,2)='b';

[~,p,~,stats] = ttest2(boots_ct_std_nc_pcorr_100.dc,boots_ct_std_ad_pcorr_100.dc);
t_dc_ct_std_nc_ad = stats.tstat;
fdr_dc_ct_std_nc_ad = fdr_bky(p);
sig_t_dc_ct_std_nc_ad = t_dc_ct_std_nc_ad(p < 0.05 & fdr_dc_ct_std_nc_ad);
sig_labels(p < 0.05 & fdr_dc_ct_std_nc_ad,3)='c';


% Plot as bar graphs (grouped)
M=[dc_ct_std_nc_boot_mean; dc_ct_std_mci_boot_mean; dc_ct_std_ad_boot_mean]';
denom=sqrt(B);
M_err=[dc_ct_std_nc_boot_stdev/denom; dc_ct_std_mci_boot_stdev/denom; dc_ct_std_ad_boot_stdev/denom]';

params=plot_horizontal_sorted_params;
params.y_labels=aal_names_76;
params.sort_order='descend';
params.plot_type='grouped';
params.plot_title='Degree centrality, sorted by NC';
params.legend_labels=[{'NC'},{'MCI'},{'AD'}];
params.max_bars=15;
params.limits=[0,1];
params.plot_size=[800 650];
plot_horizontal_sorted(M,params,M_err)
params2=params;
params2.sort_group=2;
params2.plot_title='Degree centrality, sorted by MCI';
plot_horizontal_sorted(M,params2,M_err);
params3=params;
params3.sort_group=3;
params3.plot_title='Degree centrality, sorted by AD';
plot_horizontal_sorted(M,params3,M_err);

% Plot as bar graphs (stacked)
params=plot_horizontal_sorted_params;
params.y_labels=aal_names;
params.sort_order='descend';
params.plot_type='stacked';
params.plot_title='Degree centrality, sorted by NC';
params.legend_labels=[{'NC'},{'MCI'},{'AD'}];
params.max_bars=25;
params.limits=[0,3];
params.plot_size=[800 650];
plot_horizontal_sorted(M,params)
params2=params;
params2.sort_group=2;
params2.plot_title='Degree centrality, sorted by MCI';
plot_horizontal_sorted(M,params2);
params3=params;
params3.sort_group=3;
params3.plot_title='Degree centrality, sorted by AD';
plot_horizontal_sorted(M,params3);



% Betweenness centrality
bc_ct_std_nc_fdr = betweenness_wei(1-R_ct_std_nc_fdr);
bc_ct_std_nc_fdr = normalize_data2(bc_ct_std_nc_fdr);

bc_ct_std_mci_fdr = betweenness_wei(1-R_ct_std_mci_fdr);
bc_ct_std_mci_fdr = normalize_data2(bc_ct_std_mci_fdr);

bc_ct_std_ad_fdr = betweenness_wei(1-R_ct_std_ad_fdr);
bc_ct_std_ad_fdr = normalize_data2(bc_ct_std_ad_fdr);

% Comparisons

for i = 1 : B
    boots_ct_std_nc_pcorr_100.bc(i,:) = betweenness_wei(boots_ct_std_nc_pcorr_100.r_boot(:,:,i));
    boots_ct_std_mci_pcorr_100.bc(i,:) = betweenness_wei(boots_ct_std_mci_pcorr_100.r_boot(:,:,i));
    boots_ct_std_ad_pcorr_100.bc(i,:) = betweenness_wei(boots_ct_std_ad_pcorr_100.r_boot(:,:,i));
end

boots_ct_std_nc_pcorr_100.bc = normalize_data2(boots_ct_std_nc_pcorr_100.bc);
boots_ct_std_mci_pcorr_100.bc = normalize_data2(boots_ct_std_mci_pcorr_100.bc);
boots_ct_std_ad_pcorr_100.bc = normalize_data2(boots_ct_std_ad_pcorr_100.bc);
boots_ct_std_nc_pcorr_100.bc(isnan(boots_ct_std_nc_pcorr_100.bc))=0;
boots_ct_std_mci_pcorr_100.bc(isnan(boots_ct_std_mci_pcorr_100.bc))=0;
boots_ct_std_ad_pcorr_100.bc(isnan(boots_ct_std_ad_pcorr_100.bc))=0;

% Get mean/std
bc_ct_std_nc_boot_mean = mean(boots_ct_std_nc_pcorr_100.bc,1);
bc_ct_std_mci_boot_mean = mean(boots_ct_std_mci_pcorr_100.bc,1);
bc_ct_std_ad_boot_mean = mean(boots_ct_std_ad_pcorr_100.bc,1);
bc_ct_std_nc_boot_stdev = std(boots_ct_std_nc_pcorr_100.bc,0,1);
bc_ct_std_mci_boot_stdev = std(boots_ct_std_mci_pcorr_100.bc,0,1);
bc_ct_std_ad_boot_stdev = std(boots_ct_std_ad_pcorr_100.bc,0,1);

% Get t-values
[~,p,~,stats] = ttest2(boots_ct_std_nc_pcorr_100.bc,boots_ct_std_mci_pcorr_100.bc);
t_bc_ct_std_nc_mci = stats.tstat;
fdr_bc_ct_std_nc_mci = fdr_bky(p);

[~,p,~,stats] = ttest2(boots_ct_std_mci_pcorr_100.bc,boots_ct_std_ad_pcorr_100.bc);
t_bc_ct_std_mci_ad = stats.tstat;
fdr_bc_ct_std_mci_ad = fdr_bky(p);

[~,p,~,stats] = ttest2(boots_ct_std_nc_pcorr_100.bc,boots_ct_std_ad_pcorr_100.bc);
t_bc_ct_std_nc_ad = stats.tstat;
fdr_bc_ct_std_nc_ad = fdr_bky(p);

% Plot as bar graphs (grouped)
M=[bc_ct_std_nc_boot_mean; bc_ct_std_mci_boot_mean; bc_ct_std_ad_boot_mean]';
denom=sqrt(B);
M_err=[bc_ct_std_nc_boot_stdev/denom; bc_ct_std_mci_boot_stdev/denom; bc_ct_std_ad_boot_stdev/denom]';

params=plot_horizontal_sorted_params;
params.y_labels=aal_names_76;
params.sort_order='descend';
params.plot_type='grouped';
params.plot_title='Betweenness centrality, sorted by NC';
params.legend_labels=[{'NC'},{'MCI'},{'AD'}];
params.max_bars=15;
params.limits=[0,0.2];
params.plot_size=[800 650];
plot_horizontal_sorted(M,params,M_err)
params2=params;
params2.sort_group=2;
params2.plot_title='Betweenness centrality, sorted by MCI';
plot_horizontal_sorted(M,params2,M_err);
params3=params;
params3.sort_group=3;
params3.plot_title='Betweenness centrality, sorted by AD';
plot_horizontal_sorted(M,params3,M_err);


%% 2.2.4. Output matrices to Pajek

% Load AAL data
load ( fullfile (adni_dir, 'final_data', 'aal_on_minc') );

% Create directory if necessary
pajek_dir = fullfile(output_dir, 'pajek');
if ~exist(pajek_dir, 'dir')
   mkdir(pajek_dir)
end

% Write matrices
write_matrix_to_pajek(R_ct_std_nc_fdr,fullfile(pajek_dir, 'r_pcor_ct_std_nc_fdr.net'), 'directed', false, ... 
                      'weighted', true, 'coords', aal_centroids_76, 'labels', aal_names_76);
write_matrix_to_pajek(R_ct_std_mci_fdr,fullfile(pajek_dir, 'r_pcor_ct_std_mci_fdr.net'), 'directed', false, ... 
                      'weighted', true, 'coords', aal_centroids_76, 'labels', aal_names_76);
write_matrix_to_pajek(R_ct_std_ad_fdr,fullfile(pajek_dir, 'r_pcor_ct_std_ad_fdr.net'), 'directed', false, ... 
                      'weighted', true, 'coords', aal_centroids_76, 'labels', aal_names_76);

                  
% Degree centrality
write_matrix(fullfile(pajek_dir, 'dc_pcor_ct_std_nc_fdr.txt'), dc_ct_std_nc_fdr', '\t');
write_matrix(fullfile(pajek_dir, 'dc_pcor_ct_std_mci_fdr.txt'), dc_ct_std_mci_fdr', '\t');
write_matrix(fullfile(pajek_dir, 'dc_pcor_ct_std_ad_fdr.txt'), dc_ct_std_ad_fdr', '\t');

% Betweenness centrality
write_matrix(fullfile(pajek_dir, 'bc_pcor_ct_std_nc_fdr.txt'), bc_ct_std_nc_fdr, '\t');
write_matrix(fullfile(pajek_dir, 'bc_pcor_ct_std_mci_fdr.txt'), bc_ct_std_mci_fdr, '\t');
write_matrix(fullfile(pajek_dir, 'bc_pcor_ct_std_ad_fdr.txt'), bc_ct_std_ad_fdr, '\t');


%% 2.2.5. Output t-values to Pajek

% Export difference matrices as Pajek graphs, for visualization in
% ModelGUI. Export vertex-wise data as text files.

write_matrix_to_pajek(T_nc_mci_std,fullfile(pajek_dir, 't_pcor_ct_std_nc_mci_fdr.net'), 'directed', false, ... 
                      'weighted', true, 'coords', aal_centroids_76, 'labels', aal_names_76);
write_matrix_to_pajek(T_mci_ad_std,fullfile(pajek_dir, 'r_pcor_ct_std_mci_ad_fdr.net'), 'directed', false, ... 
                      'weighted', true, 'coords', aal_centroids_76, 'labels', aal_names_76);
write_matrix_to_pajek(T_nc_ad_std,fullfile(pajek_dir, 'r_pcor_ct_std_nc_ad_fdr.net'), 'directed', false, ... 
                      'weighted', true, 'coords', aal_centroids_76, 'labels', aal_names_76);

% Degree centrality
write_matrix(fullfile(pajek_dir, 't_dc_pcor_ct_std_nc_mci_fdr.txt'), t_dc_nc_mci_fdr', '\t');
write_matrix(fullfile(pajek_dir, 't_dc_pcor_ct_std_mci_ad_fdr.txt'), t_dc_mci_ad_fdr', '\t');
write_matrix(fullfile(pajek_dir, 't_dc_pcor_ct_std_nc_ad_fdr.txt'), t_dc_mci_ad_fdr', '\t');


% Betweenness centrality
write_matrix(fullfile(pajek_dir, 't_bc_pcor_ct_std_nc_mci_fdr.txt'), t_bc_nc_mci_fdr', '\t');
write_matrix(fullfile(pajek_dir, 't_bc_pcor_ct_std_mci_ad_fdr.txt'), t_bc_mci_ad_fdr', '\t');
write_matrix(fullfile(pajek_dir, 't_bc_pcor_ct_std_nc_ad_fdr.txt'), t_bc_mci_ad_fdr', '\t');

