%% Exploring Gaussian Random Field Theory

% Generates and smooths a random 2D (statistical) image
% and illustrates how RFT can be used to analyse
% the image as a random field

% References:
%
% Blog link
% SPM textbook
% Worsley paper
%

addpath('lib/brewermap');

cmap = flipud(brewermap(60,'RdBu'));

% Size of random image in X,Y
N = 100;
M = rand(N);

% FWHM defining smoothing to apply
fwhm = [2 5 10];
Nf = length(fwhm);

close all;
h = figure;
h.Position(3:4) = [Nf*600 1500];
h.Color = 'w';

% Z-value threshold to apply to image to produce clusters
zthres = 2.5;

maxM = max(M(:)); minM = min(M(:));

for i = 1 : Nf
    
    ax = subplot(3,Nf,i);
    Mi = zscore(get_smoothed_image(M, fwhm(i)));
    
    imagesc(Mi);
    colormap(ax, cmap);
    caxis([-3 3]);
    hh = colorbar;
    hh.FontSize = 13;
    ax.XTick = [];
    ax.YTick = [];
    
    hh = title(sprintf('FWHM: %1.1f $|$ Resels: %1.0f', fwhm(i), N^2/fwhm(i)^2), 'Interpreter', 'latex');
    hh.FontSize = 25;
    
    ax = subplot(3,Nf,i+Nf);
   
    Mt = double(Mi>zthres);
    
    CC = bwconncomp(Mt);
    
    imagesc(Mt);
    colormap(ax, [[1 1 1];[.5 0 0]]);
    caxis([0 1]);
    hh = colorbar;
    hh.FontSize = 13;
    hh.Ticks = [0 1];
    
    ax.XTick = [];
    ax.YTick = [];

    hh = title(sprintf('Clusters ($t>%1.1f$): %d', zthres, CC.NumObjects), 'Interpreter', 'latex');
    hh.FontSize = 25;
    
    
end

%% Plot the Euler Characteristic (EC) as a theoretical curve (see refs)

zthres = [0:0.1:5];
C = 4 * log(2) * (2*pi)^(-3/2);

clr_1 = [.2 .2 .8];
clr_05 = [.2 .8 .2];

for f = 1 : Nf
    ax = subplot(3,Nf,f+Nf*2);
    
    R = N^2/fwhm(f)^2;
    eec = R * C * zthres .* exp(-0.5 * zthres.^2);
    idx = find(eec>1,1,'last'); if idx < length(eec), idx = idx+1; end
    idx05 = find(eec>0.025,1,'last')+1; if idx05 < length(eec), idx05 = idx05+1; end
    
    hold on;
    
    plot(zthres, eec, '-r', 'LineWidth', 1.5);
    
    ax = gca;
    ym = ax.YLim(2);
    hh = line([zthres(idx) zthres(idx)],[0 ym], 'LineWidth', 1.5);
    hh.Color = clr_1;
    ylim([0 ym]);
    
    hh = text(zthres(idx)-0.35, ym-0.1*ym, '1.0');
    hh.FontSize = 15;
    hh.Color = clr_1;
    
    zcrit = '>5.0';
    
    if eec(idx05)<=0.025
        zcrit = sprintf('=\\pm%1.1f', zthres(idx05));
        hh = line([zthres(idx05) zthres(idx05)],[0 ym], 'LineWidth', 1.5);
        hh.Color = clr_05;
        ylim([0 ym]);
        
        hh = text(zthres(idx05)-0.46, ym-0.1*ym, '.025');
        hh.FontSize = 15;
        hh.Color = clr_05;
    end
    
    ax.FontSize = 15;
    
    hh = xlabel('t threshold');
    hh.FontSize = 23;
    xtickformat('%1.1f');
    hh = ylabel('Euler characteristic');
    hh.FontSize = 23;
    
    hh = title(sprintf('$t_{crit} %s$', zcrit), 'Interpreter', 'latex');
    hh.FontSize = 25;

end

saveas(h, 'rft_figure.svg');

%% Generate EC distributions from lots of random images

Np = 1000;
zthres = [0:0.1:5];
Nz = length(zthres);

EC = zeros(Nf,Np,Nz);

for p = 1 : Np
    for f = 1 : Nf
        for i = 1 : Nz
            Mp = zscore(get_smoothed_image(rand(N),fwhm(f)));
            
            Mt = double(Mp>zthres(i));
            CC = bwconncomp(Mt);
            EC(f,p,i) = CC.NumObjects;
        end
    end
end

%% Plot distributions and see how they differ... this is because of "holes"
%  in the clusters (more holes at higher FWHM).

close all;

h = figure;
h.Position(3:4) = [1100 300];
h.Color = 'w';

zthres = [0:0.1:5];
C = 4 * log(2) * (2*pi)^(-3/2);

clr_1 = [.2 .2 .8];
clr_05 = [.2 .8 .2];

for f = 1 : Nf
    ax = subplot(1,Nf,f);
    X = squeeze(EC(f,:,:));
    Xm = mean(X,1);
    hold off;
    plot(zthres,Xm,'--r', 'LineWidth', 1.5);
    
    R = N^2/fwhm(f)^2;
    eec = R * C * zthres .* exp(-0.5 * zthres.^2);
    idx = find(eec>1,1,'last'); if idx < length(eec), idx = idx+1; end
    idx05 = find(eec>0.025,1,'last')+1; if idx05 < length(eec), idx05 = idx05+1; end
    
    hold on;
    
    plot(zthres, eec, '-r', 'LineWidth', 1.5);
    
    ax = gca;
    ym = ax.YLim(2);
    hh = line([zthres(idx) zthres(idx)],[0 ym], 'LineWidth', 1.5);
    hh.Color = clr_1;
    ylim([0 ym]);
    
    hh = text(zthres(idx)-0.35, ym-0.1*ym, '1.0');
    hh.FontSize = 15;
    hh.Color = clr_1;
    
    zcrit = '>5.0';
    
    if eec(idx05)<=0.025
        zcrit = sprintf('=\\pm%1.1f', zthres(idx05));
        hh = line([zthres(idx05) zthres(idx05)],[0 ym], 'LineWidth', 1.5);
        hh.Color = clr_05;
        ylim([0 ym]);
        
        hh = text(zthres(idx05)-0.46, ym-0.1*ym, '.025');
        hh.FontSize = 15;
        hh.Color = clr_05;
    end
    
    ax.FontSize = 15;
    
    hh = xlabel('t threshold');
    hh.FontSize = 23;
    xtickformat('%1.1f');
    hh = ylabel('Euler characteristic');
    hh.FontSize = 23;
    
    hh = title(sprintf('$t_{crit} %s$', zcrit), 'Interpreter', 'latex');
    hh.FontSize = 25;

end

