%% Explore the hemodynamic response function (HRF)
%
%


%% Imports
clear
addpath('lib/acf');

%% HRF from three gamma PDFs

X = 0:200;

theta = [1:10];

close all;
h = figure;
names = cell(length(theta),1);

for t = 1 : length(theta)
    G1 = gampdf(X, theta(t), theta(t));
    plot(X,G1);
    hold on;
    names(t) = {sprintf('%d',t)};
end

legend(names);

ylim([0 0.2]);

%% HRF is three gammas

dt = 5;
T = X/dt;
p1 = 2;
p2 = 5;
p3 = 9;

G1 = gampdf(X,p1,p1);
G1 = G1/max(G1);
G2 = gampdf(X,p2,p2);
G2 = G2/max(G2);
G3 = gampdf(X,p3,p3);
G3 = G3/max(G3);

b1 = -0.05;
b2 = 1;
b3 = -0.05;

hrf = b1*G1 + b2*G2 + b3*G3;
hrf = hrf/max(hrf);

close all;
h = figure;
h.Color = 'w';
h.Position(3:4) = [1000 500];

subplot(1,2,1);
hh = plot(T,[b1*G1;b2*G2;b3*G3], 'LineWidth', 2);
hh = title('Scaled gamma PDFs');
hh.FontSize = 18;
xlabel('time (s)');
xlim([0 25]);

subplot(1,2,2);
hh = plot(T,hrf);
hh.LineWidth = 2;
hh = title('Summed gamma PDFs');
hh.FontSize = 18;
xlabel('time (s)');
xlim([0 25]);


%% Convolution of input stimulus

Y_max = 2;

ylims = [-5 15];

N_t = 400;
X = 1:N_t;
T = X/dt;
Y = zeros(N_t,1);
S_idx = 50:51;
Y(S_idx) = 2;

Yhrf = conv(Y, hrf, 'full');
% Yhrf = Yhrf/max(Yhrf) * Y_max;

close all;
h = figure;
h.Color = 'w';
h.Position(3:4) = [1000 800];

subplot(2,2,1);
hh = plot(T,Y,'LineWidth',2);
hold on;
hh = plot(T,Yhrf(1:N_t),'LineWidth',2);
ylim(ylims);

Y = zeros(N_t,1);
S_idx = 50:100;
Y(S_idx) = 0.5;
Yhrf = conv(Y, hrf, 'full');
% Yhrf = Yhrf/max(Yhrf) * Y_max;

subplot(2,2,2);
hh = plot(T,Y,'LineWidth',2);
hold on;
hh = plot(T,Yhrf(1:N_t),'LineWidth',2);
ylim(ylims);

% Y = zeros(N_t,1);
% S_idx = 50:200;
% Y(S_idx) = 1;
Y = wgn(N_t,1,1);
Y = 0.2 * (Y-min(Y)) / max(Y);
Yhrf = conv(Y, hrf, 'full');
subplot(2,2,3);
hh = plot(T,Y,'LineWidth',2);
hold on;
hh = plot(T,Yhrf(1:N_t),'LineWidth',2);
ylim(ylims);

Y = zeros(N_t,1);
S_idx = [50:52 70:72 90:92 110:112 130:132];
Y(S_idx) = 2;
Yhrf = conv(Y, hrf, 'full');
subplot(2,2,4);
hh = plot(T,Y,'LineWidth',2);
hold on;
hh = plot(T,Yhrf(1:N_t),'LineWidth',2);
ylim(ylims);


%% HRFs plot

close all;

tick_size = 16;
label_size = 22;
title_size = 28;

dt = 5;
N_t = 800;
X = 0:0.5:N_t;
T = X/dt;
tlims = [0 25];
ylims = [-.2 1.2];

G1 = gampdf(X,p1,p1);
G1 = G1/max(G1);
G2 = gampdf(X,p2,p2);
G2 = G2/max(G2);
G3 = gampdf(X,p3,p3);
G3 = G3/max(G3);

b1 = -0.05;
b2 = 1;
b3 = -0.05;

hrf = b1*G1 + b2*G2 + b3*G3;
hrf = hrf/max(hrf);

h = figure;
h.Position(3:4) = [1600 400];
h.Color = 'w';

% 1. Gamma x 1
subplot(1,3,1);
hh = plot(T,G2, 'LineWidth', 3);
xlim(tlims);
ylim(ylims);

ax = gca;
xlabel('time (s)');
ylabel('BOLD (% change)');
ax.FontSize = label_size;

hh = title('Gamma × 1');
hh.FontSize = title_size;

% 2. Gamma x 3
subplot(1,3,2);
% hh = plot(T,[b1*G1;b2*G2;b3*G3]);
% hold on;
hh = plot(T,hrf, 'LineWidth', 3);
xlim(tlims);
ylim(ylims);

ax = gca;
xlabel('time (s)');
ylabel('BOLD (% change)');
ax.FontSize = label_size;

hh = title('Gamma × 3');
hh.FontSize = title_size;

M = [T;G1;G2;G3;hrf];
gamma_table = array2table(M','VariableNames',{'time','G1','G2','G3','Canonical'});
writetable(gamma_table, 'gamma_series.csv');

% 3. Balloon (truncate first 50 steps)
dt = 100;
N_t = 10000;
X = 0:N_t;
T = X/dt;

N_buffer = 1500;
X2 = 0:N_t+N_buffer;
T2 = X2/dt;
u = 0.01*wgn(length(T2),1,1);
u(N_buffer + [5:10]) = 20;

[tt,Y,vals] = get_balloon_sim(T2, u, [0 40]); % T2);
idx = find(tt<=T2(N_buffer),1,'last')+1;
vals = vals(idx:end,:);
tt = tt(idx:end) - T2(N_buffer);
Y = Y(idx:end);
Y = Y / max(Y);

subplot(1,3,3);
hh = plot(tt,Y, 'LineWidth', 3);
xlim(tlims);
ylim(ylims);


ax = gca;
xlabel('time (s)');
ylabel('BOLD (% change)');
ax.FontSize = label_size;
hh = title('Balloon Model');
hh.FontSize = title_size;

saveas(h, 'hrf_gamma_balloon.svg');

figure;
% idx=1:1000;
str = {'f_i_n','s','v','q'};
for i = 1 : 4
    subplot(2,2,i);
    plot(tt(:),vals(:,i));
    xlim(tlims);
    title(str{i});
end

%% Trial-base design

close all;
clear;

clrs = lines(3);

dt = 5;
X = 1:dt:200; % 1:20 seconds
p1 = 2;
p2 = 5;
p3 = 9;

G1 = gampdf(X,p1,p1);
G1 = G1/max(G1);
G2 = gampdf(X,p2,p2);
G2 = G2/max(G2);
G3 = gampdf(X,p3,p3);
G3 = G3/max(G3);

b1 = -0.05;
b2 = 1;
b3 = -0.1;

hrf = b1*G1 + b2*G2 + b3*G3;
hrf = hrf/max(hrf);

figure,plot(X/5,hrf)

dt = seconds(dt/5); 
N_trial = 10;
t_stim = seconds(0.1);
t_iti = seconds(15);

time = 0:dt:(N_trial+1)*t_iti;
N_t = length(time);
X = zeros(N_t,2);

T = randsample([ones(N_trial/2,1);2*ones(N_trial/2,1)],N_trial);
for i = 1 : N_trial
    ti = i*t_iti;
    idx1 = find(time>ti,1,'first');
    idx2 = find(time>ti+t_stim,1,'first');
    X(idx1:idx2,T(i)) = 0.5;
end

% Convolve with HRF
Y = X*0;
Y0 = Y;
for i = 1 : 2
    Yi = conv(X(:,i), hrf, 'full');
    Yi = Yi / max(Yi) * 0.5;
    Y0(:,i) = Yi(1:N_t);
    Y(:,i) = Y0(:,i) - 2.5 + i/1.5;
end

X2 = X;

% Simulate BOLD signal
Y2 = wgn(N_t,1,1,'real');
Y2 = Y2/sum(range(Y2)) / 3;
Y2 = 0.15+Y2;
Y2(Y2<0)=0;
Ynoise = conv(Y2, hrf, 'full');
Ynoise = Ynoise / max(Ynoise) * 0.70;
Ynoise = Ynoise(1:N_t);
figure;
subplot(3,1,1);
plot(Y2);
% Make faces > houses
X3=X;
X3(:,2)=X3(:,2) * 0.5;
Y2 = Y2+max(X3,[],2);
subplot(3,1,2);
plot(Y2);
Y2 = conv(Y2, hrf, 'full');
Y2 = Y2 / max(Y2) * 0.7;
k = length(hrf)-2;
Y2 = Y2(1:N_t);
subplot(3,1,3);
plot(Y2);

X(:,2) = X(:,2) + .75;

h = figure;
h.Position(3:4) = [1100 500];
h.Color = 'w';
for i = 1 : 2
    hh = plot(time, X(:,i), 'LineWidth', 1.5);
    hh.Color = clrs(i,:);
    hold on;
    hh = plot(time, Y(:,i), 'LineWidth', 1.5);
    hh.Color = clrs(i,:);
end

hh = plot(time, Y2-3.5, 'LineWidth', 1.5);
hh.Color = clrs(3,:);

grid on;

ylim([-4 1.5]);
xlim([seconds(5) time(end)]);
ax = gca;
ax.YTick = [];
ax.FontSize = 18;

hh = xlabel('Time (seconds)');
hh.FontSize = 26;

saveas(h, 'fmri_trial_ts.svg');

%% Autocorrelation function and effective degrees of freedom

% Ref: https://www.earthinversion.com/geophysics/estimation-degrees-of-freedom/

close all;
acf_Y2 = acf(Y2, 10);
idx = find(acf_Y2 < 1/exp(1),1,'first');

% Assuming idx is not empty...
t_E = seconds(dt * idx);

df_eff = (N_t-3) / (2*t_E);


%% Solve with equations
% close all;

idx0 = find(time>=seconds(5),1,'first');
idx = idx0:N_t;

clrs = lines(10);

Y = Y2(idx);
X2 = [ones(N_t-idx0+1,1),Y0(idx,:)];
Beta = X2\Y;
Ypred = X2*Beta;
Err = Y - Ypred;

h = figure;
h.Position(3:4) = [1100 500];
h.Color = 'w';

plot(time(idx), Y, 'Color', clrs(6,:), 'LineWidth', 1.5);
hold on;
plot(time(idx), Ypred, 'Color', clrs(7,:), 'LineWidth', 1.5);
plot(time(idx), Err, 'Color', clrs(4,:), 'LineWidth', 1.5);
plot(time(idx), 0.6*Ynoise(idx)-.6, 'Color', clrs(9,:), 'LineWidth', 1.5);

legend({'Y','Ypred','Err','Noise'});

ylim([-0.5 0.9]);
xlim([seconds(5) time(end)]);
ax = gca;
ax.YTick = [];
ax.FontSize = 18;

hh = xlabel('Time (seconds)');
hh.FontSize = 26;

saveas(h, 'fmri_trial_predict.svg');

% Contrast + t test
C = [0 -1 1]';
num = Beta'*C;
N = 50;
df = N/2 - 3;

sigma = sum(Err.^2)/(df_eff);
denom = sqrt(sigma * (C'*inv(X2'*X2)*C));
tval = num / denom;

% Get two-tailed p value
pval = (1-tcdf(abs(tval), df)) * 2;


%% Illustrate 2nd level analysis

dt = seconds(1); 
N_sub = 20;
N_trial = 20;

t_stim = seconds(0.1);
t_iti = seconds(15);

time = 0:dt:(N_trial+1)*t_iti;
N_t = length(time);

% Random activity magnitudes drawn from normal distribution
mu = [0.8 0.4];
sigma = [0.2 0.2];
D_sub = [(mu(1)+randn(N_sub,1)*sigma(1)),(mu(2) + randn(N_sub,1)*sigma(2))];
D_sub(D_sub>1)=1; D_sub(D_sub<0.1)=0.1;

% Relative levels
S_noise = 0.5;
S_stim = 0.8;
S_sig = 1;

X_all = zeros(N_sub,N_t,2);
Y_all = zeros(N_sub,N_t,2);
Y_noise = zeros(N_sub,N_t);

Y_max = [];
Y_max.Houses = nan(N_sub,N_trial/2);
Y_max.Faces = nan(N_sub,N_trial/2);

for s = 1 : N_sub
    
    X = zeros(N_t,2);
    T = randsample([ones(N_trial/2,1);2*ones(N_trial/2,1)],N_trial);
    for i = 1 : N_trial
        ti = i*t_iti;
        idx1 = find(time>ti,1,'first');
        idx2 = find(time>ti+t_stim,1,'first');
        X(idx1:idx2,T(i)) = 0.5;
    end

    % Convolve with HRF
    Y = X*0;
    Y0 = Y;
    for i = 1 : 2
        Yi = conv(X(:,i), hrf, 'full');
        Yi = Yi / max(Yi) * S_stim;
        Y0(:,i) = Yi(1:N_t);
        Y(:,i) = Y0(:,i);
    end

    % Simulate BOLD signal
    Y2 = wgn(N_t,1,1,'real');
    Y2 = Y2/sum(range(Y2)) / 3;
    Y2 = 0.15+Y2;
    Y2(Y2<0)=0;

    Yn = conv(Y2, hrf, 'full');
    Y_noise(s,:) = Yn(1:N_t);
    
    Y2 = Y2 * S_noise;
   
    % Make faces > houses; with random variance
    X_all(s,:,:) = Y;
    X(:,1)=X(:,1) * D_sub(s,1);
    X(:,2)=X(:,2) * D_sub(s,2);
    Y2 = Y2+max(X,[],2);
    Y2 = conv(Y2, hrf, 'full');
    Y2 = Y2 * S_sig;
    k = length(hrf)-2;
    Y2 = Y2(1:N_t);
    
%     X_all(s,:,:) = X;
    yy = conv(X(:,1), hrf, 'full');
    Y_all(s,:,1) = yy(1:N_t);
    yy = conv(X(:,2), hrf, 'full');
    Y_all(s,:,2) = yy(1:N_t);

    ym = nan(N_trial,2);
    
    % Get trial maxes
    for i = 1 : N_trial
        ti = i*t_iti;
        idx1 = find(time>ti,1,'first');
        idx2 = find(time>=ti+t_iti,1,'first');
        ym(i,T(i)) = max(Y2(idx1:idx2));
    end
    
    Y_max.Faces(s,:) = ym(~isnan(ym(:,1)),1);
    Y_max.Houses(s,:) = ym(~isnan(ym(:,2)),2);
 
end

Y_noise = Y_noise * S_noise;
Y_all = Y_all * S_sig;

%% Plot max Y faces vs. houses
close all;
h = figure;
h.Position(3:4) = [1200,500];
h.Color = 'w';
clr = rand(N_sub,3);

fmax = 1.2;

subplot(1,2,1);

line([0 fmax], [0 fmax], 'LineStyle', '--', 'LineWidth', 1.5, 'Color', [0.6 0.6 0.6]);
hold on;

for s = 1 : N_sub
    
    hh = scatter(Y_max.Houses(s,:), Y_max.Faces(s,:), 50, 'filled');
    hh.MarkerFaceColor = clr(s,:);
    
end

xlim([0.2 1]); %fmax]);
ylim([0.2 1]); % fmax]);

ax = gca;
ax.XTickLabel = [];
ax.YTickLabel = [];


xlabel('House Trials');
ylabel('Face Trials');
ax.FontSize = 18;

hh = title('Peak BOLD Intensity');
hh.FontSize = 24;

box on;
grid on;

% Plot mean + std error

subplot(1,2,2);
line([0 fmax], [0 fmax], 'LineStyle', '--', 'LineWidth', 1.5, 'Color', [0.6 0.6 0.6]);
hold on;

M = [mean(Y_max.Houses,2),mean(Y_max.Faces,2)];
S = [std(Y_max.Houses,1,2),std(Y_max.Faces,1,2)]; % / sqrt(size(Y_max.Houses,2));

hh = errorbar(M(:,1),M(:,2),S(:,1),S(:,1),S(:,2),S(:,2));
hh.LineStyle = 'none';

scatter(M(:,1),M(:,2),100,clr,'filled');

% xlim([0 fmax]);
% ylim([0 fmax]);

xlim([0.2 1]); %fmax]);
ylim([0.2 1]);

ax = gca;
ax.XTickLabel = [];
ax.YTickLabel = [];

xlabel('House Trials');
ylabel('Face Trials');
ax.FontSize = 18;

hh = title('Mean per Subject');
hh.FontSize = 24;

box on;
grid on;

saveas(h, 'fmri_2nd_level_means.svg');

%% Second-level analysis - two subjects

close all;
idx = [1 10];
Ns = length(idx);
offset = 1.5;
idx_s = 10;

clr = lines(2);

h = figure;
timei = time(idx_s:end);
timei = timei - timei(1);
Nt = length(timei);

X_stim = X_all(:,idx_s:end,:);
Y_stim = Y_all(:,idx_s:end,:);
Y_n = Y_noise(:,idx_s:end);
Y_sim = zeros(Nt,Ns);

Xd = ones(Nt*Ns,3);
Zd = zeros(Nt*Ns,Ns*2);
Yd = zeros(Nt*Ns,1);
Td = [];

for i = 1 : Ns
    Td = [Td timei];
    plot(timei, Y_stim(idx(i),:,1), 'Color', clr(i,:));
    hold on;
    plot(timei, Y_stim(idx(i),:,2)-offset, 'Color', clr(i,:));
    plot(timei, Y_n(idx(i),:)-offset*2, 'Color', clr(i,:));
    
    is = (i-1)*Nt+1;
    Xd(is:is+Nt-1,2:3) = X_stim(idx(i),:,:);
    Zd(is:is+Nt-1,(i-1)*2+(1:2)) = X_stim(idx(i),:,:);

    Y_sim(:,i) = Y_n(idx(i),:) + max(Y_stim(idx(i),:,:),[],3);
    Yd(is:is+Nt-1) = Y_sim(:,i);
    
    plot(timei, Y_sim(:,i)-offset*3, 'Color', clr(i,:));
    timei = timei+timei(end);
    
end

%% Fit with mixed-effects model

Beta = [0.25 .5 .25];
Gamma = [0 0 0 0];

Np = size(Xd,1);

Yp = Beta*Xd' + Gamma*Zd';
MSE = sum((Yp-Yd').^2) / Np;

close;
h=figure;
h.Position(3:4) = [1200 800];

subplot(2,1,1);
plot(Yp);
hold on;
plot(Yd);
title(sprintf('MSE: %1.6f', MSE));

Gamma = [.1 0 0 0];

Yp = Beta*Xd' + Gamma*Zd';
MSE = sum((Yp-Yd').^2) / Np;

subplot(2,1,2);
plot(Yp);
hold on;
plot(Yd);
title(sprintf('MSE: %1.6f', MSE));

%% Save to file

M = [Yd,Xd,Zd];
T = array2table(M, 'VariableNames', {'Y','X1','X2','X3','Z1','Z2','Z3','Z4'});

writetable(T, 'fmri_2nd_level.csv');


