function [tt,bold,yy] = get_balloon_sim(tsec, u, tspan, params)
%% Solves a Balloon model of BOLD signal generation
% See Friston et al., 2000
%
% u         Input signal
% tspan     Interval of integration (or time points)
% params    Parameters to use [defaults]:
%               params.epsilon    [0.5]
%               params.tau_s      [0.8]
%               params.tau_f      [0.4]
%               params.tau_0      [1]
%               params.alpha      [0.2]
%               params.E_0        [0.8]
%               params.V_0        [0.02]
% 

    if nargin < 4
        params = [];
        params.epsilon = .5;
        params.tau_s = 1.3;
        params.tau_f = 2.0;
        params.tau_0 = 1.9;
        params.alpha = 0.2;
        params.E_0 = 0.8;
        params.V_0 = 0.02;
    end

    [tt,yy] = ode45(@(t,y) ode_balloon(t,y,tsec,u), tspan, [0 0 params.V_0 0]);

    bold = lambda(yy(:,3), yy(:,4));


    function f = f_out(v)
        f = v^(1/params.alpha);
    end

    function y = lambda(v, q)
        k1 = 7*params.E_0;
        k2 = 2;
        k3 = 2*params.E_0 - 0.2;
        y = params.V_0 * ((k1*(1-q)) + k2*(1-q./v) + k3*(1-v));
    end

    function efrac = E(f_in)
        efrac = 1 - (1-params.E_0)^(1/f_in);
    end

    % y(1) = f_in
    % y(2) = s
    % y(3) = v
    % y(4) = q
    function dydt = ode_balloon(t,y, tsec, u)
        
        f_in =  y(1);
        s =     y(2);
        v =     y(3);
        q =     y(4);
        
        dydt(1) = s;
%         idx = find((tspan-t)<=0,1,'last');
        uu = interp1(tsec, u, t);
        dydt(2) = params.epsilon * uu - s/params.tau_s - (f_in - 1)/params.tau_f;
        dydt(3) = (f_in - f_out(v)) / params.tau_0;
        dydt(4) = (f_in * E(f_in) / params.E_0 - f_out(v) * q / v) / params.tau_0;
        
        dydt = dydt(:);
    end

end

