%% Run once
%addpath(fullfile(pwd,'..','anneal'));
%addpath(fullfile(pwd,'..','lib','export_fig'));

%% Create a simple forage environment and do simulated annealing on it

gaus1 = fspecial('gaussian', 40, 7)*2;
gaus2 = fspecial('gaussian', 40, 7).*2;
gaus3 = -fspecial('gaussian', 70, 10).*3;
gaus4 = -fspecial('gaussian', 100, 15).*5;

E = zeros(100);

E(6:45,6:45) = gaus1;
E(51:90,41:80) = gaus2;
E(65:100,1:30) = gaus3(1:36,31:60);
E(1:40,61:100) = gaus4(46:85,11:50);
E = imgaussfilt(E,3);

% Add some white noise
R = (randn(100)*0.5)+0.5;
R=imgaussfilt(R,3);
R=imgaussfilt(R,3);
R=(R-min(R(:)))/100;
E=E+R;

z_scale = [min(E(:)) max(E(:))];

h = figure; 
hh = imagesc(E);
caxis(z_scale);

title('Foraging environment');
xlabel('x');
ylabel('y');


%% Set up annealing

run_name = 'fixed_solution';
%run_name = 'fixed_solution_lowT';

XY0 = [ceil(rand*size(E,1)) ceil(rand*size(E,1))];
iter = 1000;
v = [1 1];
T_range = [0.01 0.05];  % Annealing
%T_range = [0.001 0.001];  % Fixed low T
N_pts = 20;

Sa = zeros(N_pts,iter,2);
Acc = zeros(N_pts,iter);
T = zeros(N_pts,iter);

for i = 1 : N_pts
    v = [0.5 + 0.5*rand 0.5 + 0.5*rand];
    XY0 = [ceil(rand*size(E,1)) ceil(rand*size(E,2))];
    [ sa, acc, t ] = anneal2d( -E, XY0, T_range, iter, v );
    
    Sa(i,:,:) = sa;
    Acc(i,:) = acc;
    T(i,:) = t;
    
end

%% Plot x,y

y_value_limits = [0 0.14];
y_temp_limits = [0 50];

h = figure;
set(h,'Color','w');

hh = line(1:iter,t*1000);
set(hh,'LineWidth',2);
set(hh,'Color','b');

hold on;

V = zeros(iter,1);
for i = 1 : iter
    si = squeeze(Sa(:,i,:));
    idx = sub2ind(size(E),si(:,1),si(:,2));
    V(i) = sum(E(idx));
end

ax1 = gca;
ax1_pos = ax1.Position; % position of first axes
ax2 = axes('Position',ax1_pos,...
            'XAxisLocation','bottom',...
            'YAxisLocation','right',...
            'Color','none');
hh = line(1:iter,V,'Parent',ax2);

set(ax1,'YLim',y_temp_limits);
set(ax2,'YLim',y_value_limits);

set(hh,'LineWidth',2);
set(hh,'Color','r');

set(ax1,{'ycolor'},{'b'});
set(ax2,{'ycolor'},{'r'});
set(ax1,'YTickLabels','');
set(ax1,'XTickLabels','');
set(ax2,'YTickLabels','');
set(ax2,'XTickLabels','');

axes(ax1);
hl = ylabel('Temperature');
set(hl,'Color','b');
set(hl,'FontSize',20);
axes(ax2);
hl = ylabel('Total value','rot',-90);
set(hl,'FontSize',20);
set(hl,'Position',get(hl,'Position') + [50 0 0])
set(hl,'Color','r');

hl = xlabel('Time');
set(hl,'Color','k');
set(hl,'FontSize',20);

export_fig(h,sprintf('time_temp_value_%s.png',run_name),'-nocrop');


%% Movie

output_dir = run_name;
tofile = false;
frames = 250;

if ~isempty(output_dir) > 0
   if exist(output_dir,'dir')
       rmdir(output_dir,'s');
   end
   mkdir(output_dir);
   filename = fullfile(output_dir,'movie');
   tofile = true;
end

h = figure; 
set(h,'Color','w');
imagesc(E');
ht = title('Foraging the environment');
set(ht,'FontSize',20);
hold on;

hh = scatter(squeeze(Sa(:,1,1)),squeeze(Sa(:,1,2)),100,'r','filled');

plotx = squeeze(Sa(:,1,1));
ploty = squeeze(Sa(:,1,2));

set(hh,'XDataSource','plotx');
set(hh,'YDataSource','ploty');

set(gca,'XTickLabel','');
set(gca,'YTickLabel','');

plottemp.pos=[90,-5];
plottemp.t=t;
plottemp.size=15;
plottemp.color=[0 0 1];
h_txt=text(plottemp.pos(1),plottemp.pos(2),sprintf('T=%1.0f',plottemp.t(1)*1000));
set(h_txt,'FontSize',plottemp.size);
set(h_txt,'Color',plottemp.color);

fnitr = 1; fileitr = 1; step = 1;
if length(t) > frames
    step = round(length(t) / frames);
end

for i = 1 : length(t)
   
    plotx = squeeze(Sa(:,i,1));
    ploty = squeeze(Sa(:,i,2));
    set(h_txt,'String',sprintf('T=%1.0f',plottemp.t(i)*1000)); 
    refreshdata;
    drawnow;
    
    if tofile && fnitr==step
        export_fig(gcf,sprintf('%s.%05d.png',filename,fileitr),'-nocrop');
        fileitr = fileitr + 1;
        fnitr=1;
    else
        fnitr=fnitr+1;
    end
    
end


%% Predators

Ep = zeros(100);

gaus5 = -fspecial('gaussian', 200, 40).*10;

Ep(1:70,1:70) = gaus5(131:200,131:200);
Ep = imgaussfilt(Ep,3)*20;

h=figure; imagesc(-Ep);
set(h,'Color','w');
set(gca,'XTickLabel','');
set(gca,'YTickLabel','');
%caxis(z_scale);
ht = title('Predator encroachment');
set(ht,'FontSize',20);

export_fig(h,'predator_encroachment.png','-nocrop');

f = 0.5;
Enew = ((E*(1-f))+Ep*f)*2;

h=figure; imagesc(Enew);
set(h,'Color','w');
caxis(z_scale);
set(gca,'XTickLabel','');
set(gca,'YTickLabel','');

ht = title('Predator encroachment');
set(ht,'FontSize',20);


export_fig(h,'predator_encroachment_value.png','-nocrop');

%% Re-explore

%XY0 = [ceil(rand*size(E,1)) ceil(rand*size(E,1))];
iter = 500;
%v = [1 1];
T_range = [0.05 0.01];
%N_pts = 20;

Sa_pred = zeros(N_pts,iter*2,2);
Acc_pred = zeros(N_pts,iter*2);
T_pred = zeros(N_pts,iter*2);

for i = 1 : N_pts
    v = [0.5 + 0.5*rand 0.5 + 0.5*rand];
    XY0 = squeeze(Sa(i,end,:));
    [ sa, acc, t ] = anneal2d( -Enew, XY0, T_range, iter, v );
    
    Sa_pred(i,1:iter,:) = sa;
    Acc_pred(i,1:iter) = acc;
    T_pred(i,1:iter) = t;
    
end

T_range = [0.01 0.05];
for i = 1 : N_pts
    v = [0.5 + 0.5*rand 0.5 + 0.5*rand];
    XY0 = squeeze(Sa_pred(i,iter,:));
    [ sa, acc, t ] = anneal2d( -Enew, XY0, T_range, iter, v );
    
    Sa_pred(i,iter+1:end,:) = sa;
    Acc_pred(i,iter+1:end) = acc;
    T_pred(i,iter+1:end) = t;
    
end

%% Plot x,y

y_value_limits = [0 0.18];
y_temp_limits = [0 50];

h = figure;
set(h,'Color','w');

iter2 = iter*2;

t = T_pred(1,:);

hh = line(1:iter2,t*1000);
set(hh,'LineWidth',2);
set(hh,'Color','b');

hold on;

V = zeros(iter2,1);
for i = 1 : iter2
    si = squeeze(Sa_pred(:,i,:));
    idx = sub2ind(size(Enew),si(:,1),si(:,2));
    V(i) = sum(Enew(idx));
end

ax1 = gca;
ax1_pos = ax1.Position; % position of first axes
ax2 = axes('Position',ax1_pos,...
            'XAxisLocation','bottom',...
            'YAxisLocation','right',...
            'Color','none');
hh = line(1:iter2,V,'Parent',ax2);

set(ax1,'YLim',y_temp_limits);
set(ax2,'YLim',y_value_limits);

set(hh,'LineWidth',2);
set(hh,'Color','r');

set(ax1,{'ycolor'},{'b'});
set(ax2,{'ycolor'},{'r'});
set(ax1,'YTickLabels','');
set(ax1,'XTickLabels','');
set(ax2,'YTickLabels','');
set(ax2,'XTickLabels','');

axes(ax1);
hl = ylabel('Temperature');
set(hl,'Color','b');
set(hl,'FontSize',20);
axes(ax2);
hl = ylabel('Total utility','rot',-90);
set(hl,'FontSize',20);
set(hl,'Position',get(hl,'Position') + [50 0 0])
set(hl,'Color','r');

hl = xlabel('Time');
set(hl,'Color','k');
set(hl,'FontSize',20);

export_fig(h,sprintf('time_temp_pred_value_%s.png',run_name),'-nocrop');

%% Movie

output_dir = 'pred_value';
tofile = false;
frames = 500;

if ~isempty(output_dir) > 0
   if exist(output_dir,'dir')
       rmdir(output_dir,'s');
   end
   mkdir(output_dir);
   filename = fullfile(output_dir,'movie');
   tofile = true;
end

h = figure; 
imagesc(Enew');
ht = title('Adapting to new predators');
set(ht,'FontSize',20);
hold on;

set(h,'Color','w');
set(gca,'XTickLabel','');
set(gca,'YTickLabel','');

plottemp.pos=[90,-5];
plottemp.t=t;
plottemp.size=15;
plottemp.color=[0 0 1];
h_txt=text(plottemp.pos(1),plottemp.pos(2),sprintf('T=%1.0f',plottemp.t(1)*1000));
set(h_txt,'FontSize',plottemp.size);
set(h_txt,'Color',plottemp.color);

hh = scatter(squeeze(Sa(:,1,1)),squeeze(Sa(:,1,2)),100,'r','filled');

plotx = squeeze(Sa_pred(:,1,1));
ploty = squeeze(Sa_pred(:,1,2));

set(hh,'XDataSource','plotx');
set(hh,'YDataSource','ploty');

fnitr = 1; fileitr = 1; step = 1;
if length(t) > frames
    step = round(length(t) / frames);
end

for i = 1 : size(T_pred,2)
   
    ti = T_pred(1,i);
    plotx = squeeze(Sa_pred(:,i,1));
    ploty = squeeze(Sa_pred(:,i,2));
    refreshdata;
    drawnow;
    
    set(h_txt,'String',sprintf('T=%1.0f',ti*1000)); 
    refreshdata;
    drawnow;
    
    if tofile && fnitr==step
        export_fig(gcf,sprintf('%s.%05d.png',filename,fileitr),'-nocrop');
        fileitr = fileitr + 1;
        fnitr=1;
    else
        fnitr=fnitr+1;
    end
    
end


%% Adapt to predators with selective T increase

% Scale temp by Ep
iter = 1000;

Sa_pred2 = zeros(N_pts,iter,2);
Acc_pred2 = zeros(N_pts,iter);
T_pred2 = zeros(N_pts,iter);

T_range = [0.013 0.013];
S=ones(100)-(Ep*1300);
for i = 1 : N_pts
    v = [0.5 + 0.5*rand 0.5 + 0.5*rand];
    XY0 = squeeze(Sa(i,iter,:));
    [ sa, acc, t ] = anneal2d_scale( -E, XY0, T_range, iter, v, S );
    
    Sa_pred2(i,:,:) = sa;
    Acc_pred2(i,:) = acc;
    T_pred2(i,:) = t;
    
end

%% Plot x,y

y_value_limits = [0 0.18];
y_temp_limits = [0 50];

h = figure;
set(h,'Color','w');

iter2 = iter*2;

t = mean(T_pred2,1)';

hh = line(1:iter2,t*1000);
set(hh,'LineWidth',2);
set(hh,'Color','b');

hold on;

V = zeros(iter2,1);
for i = 1 : iter2
    si = squeeze(Sa_pred2(:,i,:));
    idx = sub2ind(size(Enew),si(:,1),si(:,2));
    V(i) = sum(Enew(idx));
end

ax1 = gca;
ax1_pos = ax1.Position; % position of first axes
ax2 = axes('Position',ax1_pos,...
            'XAxisLocation','bottom',...
            'YAxisLocation','right',...
            'Color','none');
hh = line(1:iter2,V,'Parent',ax2);

set(ax1,'YLim',y_temp_limits);
set(ax2,'YLim',y_value_limits);

set(hh,'LineWidth',2);
set(hh,'Color','r');

set(ax1,{'ycolor'},{'b'});
set(ax2,{'ycolor'},{'r'});
set(ax1,'YTickLabels','');
set(ax1,'XTickLabels','');
set(ax2,'YTickLabels','');
set(ax2,'XTickLabels','');

axes(ax1);
hl = ylabel('Temperature (mean)');
set(hl,'Color','b');
set(hl,'FontSize',20);
axes(ax2);
hl = ylabel('Total utility','rot',-90);
set(hl,'FontSize',20);
set(hl,'Position',get(hl,'Position') + [50 0 0])
set(hl,'Color','r');

hl = xlabel('Time');
set(hl,'Color','k');
set(hl,'FontSize',20);

export_fig(h,sprintf('time_seltemp_pred_value_%s.png',run_name),'-nocrop');


%% Movie

output_dir = 'pred_seltemp_value';
tofile = false;
frames = 500;

if ~isempty(output_dir) > 0
   if exist(output_dir,'dir')
       rmdir(output_dir,'s');
   end
   mkdir(output_dir);
   filename = fullfile(output_dir,'movie');
   tofile = true;
end

h = figure; 
imagesc(Enew');
ht = title('Selective adaptation to predators');
set(ht,'FontSize',20);
hold on;

set(h,'Color','w');
set(gca,'XTickLabel','');
set(gca,'YTickLabel','');

t = mean(T_pred2,1);

plottemp.pos=[90,-5];
plottemp.t=t;
plottemp.size=15;
plottemp.color=[0 0 1];
h_txt=text(plottemp.pos(1),plottemp.pos(2),sprintf('%s=%1.0f','$\bar{T}$',plottemp.t(1)*1000));
set(h_txt,'interpreter','Latex');
set(h_txt,'FontSize',plottemp.size);
set(h_txt,'Color',plottemp.color);

hh = scatter(squeeze(Sa_pred2(:,1,1)),squeeze(Sa_pred2(:,1,2)),100,'r','filled');

plotx = squeeze(Sa_pred2(:,1,1));
ploty = squeeze(Sa_pred2(:,1,2));

set(hh,'XDataSource','plotx');
set(hh,'YDataSource','ploty');

fnitr = 1; fileitr = 1; step = 1;
if length(t) > frames
    step = round(length(t) / frames);
end

for i = 1 : size(T_pred2,2)
   
    plotx = squeeze(Sa_pred2(:,i,1));
    ploty = squeeze(Sa_pred2(:,i,2));
    refreshdata;
    drawnow;
    
    set(h_txt,'String',sprintf('%s=%1.0f','$\bar{T}$',t(i)*1000)); 
    refreshdata;
    drawnow;
    
    if tofile && fnitr==step
        export_fig(gcf,sprintf('%s.%05d.png',filename,fileitr),'-nocrop');
        fileitr = fileitr + 1;
        fnitr=1;
    else
        fnitr=fnitr+1;
    end
    
end

