function comet_anneal(varargin)
%COMET  Comet-like trajectory.
%   COMET(Y) displays an animated comet plot of the vector Y.
%   COMET(X,Y) displays an animated comet plot of vector Y vs. X.
%   COMET(X,Y,p) uses a comet of length p*length(Y).  Default is p = 0.10.
%
%   COMET(AX,...) plots into AX instead of GCA.
%
%   Example:
%       t = -pi:pi/200:pi;
%       comet(t,tan(sin(t))-sin(tan(t)))
%
%   See also COMET3.

%   Charles R. Denham, MathWorks, 1989.
%   Revised 2-9-92, LS and DTP; 8-18-92, 11-30-92 CBM.
%   Copyright 1984-2007 The MathWorks, Inc.

% Parse possible Axes input
[ax,args,nargs] = axescheck(varargin{:});

error(nargchk(1,6,nargs,'struct'));
itrp=0;
fn=[]; plottemp=[];

% Parse the rest of the inputs
if nargs < 2, x = args{1}; y = x; x = 1:length(y); end
if nargs == 2, [x,y] = deal(args{:}); end
if nargs < 3, p = 0.10; end
if nargs == 3, [x,y,p] = deal(args{:}); end
if nargs == 4, [x,y,p,itrp] = deal(args{:}); end
if nargs == 5, [x,y,p,itrp,fn] = deal(args{:}); end
if nargs == 6, [x,y,p,itrp,fn,plottemp] = deal(args{:}); end

tofile = ~isempty(fn);
showtime = ~isempty(plottemp);

if tofile
   if length(y)>fn.frames;
      step = round(length(y) / fn.frames);
   end
end

if ~isscalar(p) || ~isreal(p) ||  p < 0 || p >= 1
    error(message('MATLAB:comet:InvalidP'));
end

ax = newplot(ax);
if ~ishold(ax)
    [minx,maxx] = minmax(x);
    [miny,maxy] = minmax(y);
    axis(ax,[minx maxx miny maxy])
end
co = get(ax,'colororder');

if showtime
   h_txt=text(plottemp.pos(1),plottemp.pos(2),sprintf('T=%1.3f',plottemp.t(1)));
   set(h_txt,'FontSize',plottemp.size);
   set(h_txt,'Color',plottemp.color);
end

m = length(x);
k = round(p*m);

if size(co,1)>=3
    colors = [ co(1,:);co(2,:);co(3,:)];
    lstyle = '-';
else
    colors = repmat(co(1,:),3,1);
    lstyle = '--';
end

% head = line('parent',ax,'color',colors(1,:),'marker','o','MarkerSize',10, ...
%             'MarkerFaceColor', colors(1,:), 'linestyle','none', ...
%             'xdata',x(1),'ydata',y(1),'Tag','head');
        
[m,n] = size(x);

for i=1:n
    % Choose first three colors for head, body, and tail
    head(i) = line('parent',ax,'color',colors(1,:),'marker','o','MarkerSize',10, ...
                    'MarkerFaceColor', colors(1,:), 'linestyle','none', ...
                    'xdata',x(1,i),'ydata',y(1,i),'Tag',sprintf('head%d',i));
end
        
if ( length(x) < 2000 )
    updateFcn = @()drawnow;
else
    updateFcn = @()drawnow('update');
end

% Grow the body
fnitr=1;
fileitr=1;

% Add a drawnow to capture any events / callbacks
drawnow;
% Primary loop
for i = 1:m
    pause(itrp);
    for a = 1:n
        set(head(a),'xdata',x(i,a),'ydata',y(i,a));
    end
    if showtime
        set(h_txt,'String',sprintf('T=%1.3f',plottemp.t(i))); 
    end
    updateFcn();
    if tofile && fnitr==step
        export_fig(gcf, sprintf('%s.%05d.png',fn.filename,fileitr));
        fileitr = fileitr + 1;
        fnitr=1;
    else
        fnitr=fnitr+1;
    end
end

drawnow;
end


function [minx,maxx] = minmax(x)
minx = min(x(isfinite(x)));
maxx = max(x(isfinite(x)));
if minx == maxx
    minx = maxx-1;
    maxx = maxx+1;
end
end
