function h = plot_g2d( x, y, peaks, fwhm, window_size, n_perms, alpha, show_titles )

    if nargin < 7
       alpha = 0.05; 
    end
    
    if nargin < 8
       show_titles = true;
    end

    xy = [x(peaks(:,1));y(peaks(:,2))];
    
    p_base = size(peaks,1) / (length(x) * length(y));

    h = figure;
    h.Color = 'w';
    
    hh = scatter(xy(1,:), xy(2,:), 'filled');
    hh.MarkerFaceColor = 'k';
    
    xlim([min(x) max(x)]);
    ylim([min(y) max(y)]);
    
    if show_titles
        hh = title('Observed foci');
        hh.FontSize = 16;
    end
    resize_window(h, window_size);
    axis('off');
    
    sigma = 2.355*fwhm;  % Some scaling
    
    % Gaussian
    M = zeros(length(x), length(y));
    for p = 1 : length(peaks)
       G = gauss2d(M,2,sigma,peaks(p,:));
       M = M+G;
    end
    
    M = (M / length(peaks)) - p_base;
    M_max = max(M(:));
    M = M';
    M = flip(M,1);
    h = figure;
    h.Color = 'w';
    hh = imagesc(M);
    colormap('jet');
    caxis([0,M_max]);
    colorbar;
    
    if show_titles
        hh = title('ALE map of observed foci');
        hh.FontSize = 16;
    end
    resize_window(h, window_size);
    axis('off');
    
    % Random Gaussian
    Mp = zeros(length(x), length(y));
    pperm = [randsample(length(x),size(peaks,1)),randsample(length(y),size(peaks,1))];
    
    % Dots
    h = figure;
    h.Color = 'w';
    
    xyp = [x(pperm(:,1));y(pperm(:,2))];
    hh = scatter(xyp(1,:), xyp(2,:), 'filled');
    hh.MarkerFaceColor = 'k';
    
    xlim([min(x) max(x)]);
    ylim([min(y) max(y)]);
    
    if show_titles
        hh = title('Permuted foci');
        hh.FontSize = 16;
    end
    resize_window(h, window_size);
    axis('off');
    
    % Image
    for p = 1 : length(pperm)
       Mp = Mp + gauss2d(Mp,2,sigma,pperm(p,:));
    end
    
    Mp = (Mp / length(peaks)) - p_base;
    Mp = Mp';
    Mp = flip(Mp,1);
    h = figure;
    h.Color = 'w';
    hh = imagesc(Mp);
    colormap('jet');
    caxis([0,M_max]);
    colorbar;
    
    ax = gca;
    ax.XAxis.TickLabels=[];
    ax.YAxis.TickLabels=[];
    
    if show_titles
        hh = title('ALE map of permuted foci');
        hh.FontSize = 16;
    end
    resize_window(h, window_size);
    axis('off');
 
    % Histogram
    Mmax = zeros(n_perms,1);
    for i = 1 : n_perms
        Mp = zeros(length(x), length(y));
        pperm = [randsample(length(x),size(peaks,1)),randsample(length(y),size(peaks,1))];
        for p = 1 : length(pperm)
           Mp = Mp + gauss2d(Mp,2,sigma,pperm(p,:));
        end
        Mp = (Mp / length(peaks)) - p_base;
        Mmax (i) = max(Mp(:));
    end
    
    [hhist,hbins] = hist(Mmax,30);
    
    % Smooth a bit
    hhist = smooth(hhist);
    
    hhist = hhist / trapz(hhist);
    cdfhist = hhist * 0;
    idx_crit = 0;
    for i = 2 : length(hhist)
        cdfhist(i) = trapz(hhist(1:i));
        if cdfhist(i) < (1-alpha)
            idx_crit = i;
        end
    end
    
    h = figure;
    h.Color = 'w';
    yyaxis left;
    
    hh = plot(hbins,hhist,'b');
    hold on;
    hh.LineWidth = 2;
    % Draw alpha% area
    if idx_crit > 0
        hh = area(hbins(idx_crit:end),hhist(idx_crit:end));
        hh.FaceColor = 'b';
        hh.FaceAlpha = 0.4;
    end
    
    if show_titles
        ylabel('Probability density');
    end
    yyaxis right;
    hh = plot(hbins,cdfhist,'r');
    hh.LineWidth = 2;
    hold on;
    if idx_crit > 0
        hh = line([hbins(idx_crit) hbins(idx_crit)],[-0 1],'LineStyle','--');
        hh.LineWidth=1.5;
        hh.Color='k';
    end
    
    if show_titles
        ax = gca;
        ax.FontSize = 14;
        hh = ylabel('Cumulative probability');
        hh.Rotation = -90;
        hh.Units = 'Normalized';
        hh.Position(1) = 1.09;
        xlabel('Maximal ALE value');
        hh = title('Non-parametric null distribution');
        hh.FontSize = 16;
    end
    resize_window(h, window_size);
    
    fprintf('Critical value (p<%1.2f): %1.4f\n', alpha, hbins(idx_crit));
    
    % Thresholded
    Mt = M;
    Mt(M < hbins(idx_crit)) = 0;
    
    h = figure;
    h.Color = 'w';
    hh = imagesc(Mt);
    colormap('jet');
    caxis([0,M_max]);
    colorbar;
    
    if show_titles
        hh = title('Thresholded ALE map');
        hh.FontSize = 16;
    end
    
    resize_window(h, window_size);
    axis('off')
    
end

