%% 1D Gaussian

% This section generates a simple 1D example of applying a Gaussian kernel
% to foci. This is the principle underlying Activation Likelihood
% Estimation (ALE).

x = 0:0.02:2;
peaks = [0.2 0.5 0.6 0.67 1.2 1.3 1.6 1.95];
peaks = round(peaks*length(x)/max(x));

plot_g1d(x, peaks, 0.2, [800 400]);

%% 2d Gaussian

% This section generates a 2D example of the ALE permutation procedure. 
% 200 points are randomly generated, with bias in particular quadrants of 
% the 2D space -- simulating a real underlying activation pattern. These points represent
% activation foci in 2D space. 

% The following plots are generated: 

% 1. Scatterplot of biased random 2D points, representing a set of foci
%    retrieved from BrainMap (www.brainmap.org)
% 2. A Gaussian field constructed from the foci in (1), representing the
%    likelihood estimate (ALE) of finding a foci at each pixel
% 3. Scatterplot of unbiased random 2D points, representing a single
%    permutation sample from which to generate a null distribution
% 4. A Gaussian field constructed from the foci in (3)
% 5. A histogram (blue curve) of the maximal ALE values across all 
%    permutations. This represents a PDF of the occurrence of particular 
%    maxima. A CDF is also shown (red curve). The vertical line shows the 
%    critical value at p<alpha; the shaded area under the curve is equal to 
%    alpha.


x = 0:0.01:2;
y = x;

show_titles = true;  % Set this to false to hide the titles and axes labels

widthx = 200; widthy = 200; % The width of the field in a.u.
fwhm = 15;                  % Full width at half-max of the Gaussian kernel
N_perm = 1000;              % Number of permutations to generate
alpha = 0.05;               % Critical p-value

fwhm = fwhm/((widthx+widthy)/(2*x(end))); % Scale FWHM to [0 max(x)];

% Generate some random foci
peaks = [1+rand(20,1),1.5+rand(20,1)*.3];
peaks = [peaks;1.3+rand(20,1)*0.5,1.5+rand(20,1)*.3];
peaks = [peaks;[0.5+rand(20,1)*0.2,rand(20,1)*.5]];
peaks = [peaks;rand(20,2)*2];
peaks = [ceil(peaks(:,1)*length(x)/max(x)) ceil(peaks(:,2)*length(y)/max(y))];

plot_g2d(x, y, peaks, fwhm, [700 550], 1000, alpha, show_titles);

